/**
 * @file RunningState.h
 * @brief The header file for the RunningState class.
 * @author Jordan Brack <jabrack@mix.wvu.edu>, Haofan Zheng <hazheng@mix.wvu.edu>
 * 
 */
#pragma once
#ifndef RUNNING_STATE_H
#define RUNNING_STATE_H

#include <mbed.h>
#include "States.h"

/**
 * @class RunningState
 * @brief The class for running state. It is a child class of the States class.
 * 
 */
class RunningState : public States
{
public:
    /**
    * @brief This is the constructor for the RunningState class
    * 
    */
    RunningState();
    /**
    * @brief This is the destructor for the RunningState class
    * 
    */
    ~RunningState();
    
    /**
    * @brief Draw the user interface. This function is called only once, whenever the running state becomes the current state.
    * 
    */
    virtual void DrawUserInterface();
    
    /**
    * @brief This functino will be called during every tick, only if the running state is the current state. The image processing, path selection, and driving decision will run from here.
    * @param deltaTime The time interval between last tick and current tick.
    * 
    */
    virtual void Update(float deltaTime);
    
    /**
    * @brief Return wether or not the running state has a position callback function for the touch screen.
    * @return 0 - not. Because the running state does not have a position callback function for the touch screen.
    * 
    */
    virtual uint8_t HasTouchPosFunction() const;
    
    /**
    * @brief Return wether or not the running state has a interrupt callback function for the touch screen.
    * @return 1 - yes. Because the running state does have a interrupt callback function for the touch screen.
    * 
    */
    virtual uint8_t HasTouchIrqFunction() const;
    
    /**
    * @brief The interrupt callback function for the touch screen.
    * 
    */
    virtual void TouchIrqCallback();
    
private:
    
    float m_speedRatio; /*!< @brief The percentage of the speed(power) that the motor driver need to maintain during a curve. */
    float m_lastAngle; /*!< @brief The final calculated turn angle from last tick. */
    float m_lastAngleAbs; /*!< @brief The final calculated turn angle from last tick, in absolute value. */
    float m_cornerRatio; /*!< @brief The percentage of the speed(power) that the motor driver need to maintain when enter/exit a curve. */
    //uint8_t m_lastLineFound; /*!< @brief  */
    volatile uint8_t m_shouldTerminate; /*!< @brief Did running state received a pendding Terminating command? */
    
    /**
    * @brief The terminating process. Called when it is necessary to terminate the runnning state. And it will switch to the standby state.
    * 
    */
    void TerminateProcess();
};

#endif //RUNNING_STATE_H