
// LED Strip using WS2801 with two ISRs for two SPI connected 
// LED strips running in parallel
// Rob Dobson 2013-2014

#include "ledstrip.h"
#include "colourconverters.h"
#include "stddef.h"

#define SPIF 0              // SPI interrupt flag bit
#define SSP_IMSC_TX_RDY 3
#define SSP_IMSC_BITMASK 0x0f

volatile int mCurPos0 = 0;
int mEndPos0 = 0;
volatile bool isr0Busy = false;
unsigned char* pLedValues0 = NULL;

extern "C" void spi0_isr()
{
    if (mCurPos0 < mEndPos0)
    {
        LPC_SSP0->DR = pLedValues0[mCurPos0];  // write to FIFO data register
        mCurPos0++;
    }
    else
    {
        // Turn off interrupts
        LPC_SSP0->IMSC = 0;
        isr0Busy = false;
    }
}

volatile int mCurPos1 = 0;
int mEndPos1 = 0;
volatile bool isr1Busy = false;
unsigned char* pLedValues1 = NULL;

extern "C" void spi1_isr()
{
    if (mCurPos1 < mEndPos1)
    {
        LPC_SSP1->DR = pLedValues1[mCurPos1];  // write to FIFO data register
        mCurPos1++;
    }
    else
    {
        // Turn off interrupts
        LPC_SSP1->IMSC = 0;
        isr1Busy = false;
    }
}

ledstrip::ledstrip(int length, int splitPoint)
{
    mpLedValuesA = NULL;
    mpLedValuesB = NULL;
    mpCurLedValues = NULL;

    // SPI0 (using SSP 0 in 1768 chip)
    mpSPI0 = new SPI(p11, NC, p13);
    mpSPI0->format(8,0);
    mpSPI0->frequency(500000);    
    LPC_SSP0->IMSC = 0; // initially no interrupts requested
    NVIC_SetVector(SSP0_IRQn,( uint32_t ) spi0_isr);
    NVIC_ClearPendingIRQ(SSP0_IRQn);
    NVIC_SetPriority(SSP0_IRQn, 2);
    NVIC_EnableIRQ(SSP0_IRQn);

    // SPI1 (using SSP 1 in 1768 chip)
    mpSPI1 = new SPI(p5, NC, p7);
    mpSPI1->format(8,0);
    mpSPI1->frequency(500000);    
    LPC_SSP1->IMSC = 0; // initially no interrupts requested
    NVIC_SetVector(SSP1_IRQn,( uint32_t ) spi1_isr);
    NVIC_ClearPendingIRQ(SSP1_IRQn);
    NVIC_SetPriority(SSP1_IRQn, 2);
    NVIC_EnableIRQ(SSP1_IRQn);

    // Resize the string length
    Resize(length, splitPoint);
}

ledstrip::~ledstrip()
{
    delete mpLedValuesA;
    delete mpLedValuesB;
}

bool ledstrip::Resize(int length, int splitPoint)
{
    if (isr0Busy || isr1Busy)
        return false;
    if (mpLedValuesA != NULL)
        delete mpLedValuesA;
    if (mpLedValuesB != NULL)
        delete mpLedValuesB;
    mLedsBufSize = length*mColoursPerLed;
    mpLedValuesA = new unsigned char[mLedsBufSize];
    mpLedValuesB = new unsigned char[mLedsBufSize];
    mpCurLedValues = mpLedValuesA;
    mLedsInStrip = length;
    mSplitPoint = splitPoint;
    Clear();
    return true;
}

void ledstrip::Clear()
{
/*    Timer timr;
    timr.start();
    for (int i = 0; i < mLedsInStrip*mColoursPerLed; i++)
        mpCurLedValues[i] = 0;
    timr.stop();
    printf("ClearTime loop %d\n", timr.read_us());  // Result is 863uS for 2500 x 3colour LEDS
    timr.reset();
    timr.start();
 */
    memset(mpCurLedValues, 0, mLedsBufSize);
 /*   timr.stop();
    printf("ClearTime memset %d\n", timr.read_us());  // Result is 35uS for 2500 x 3 colour LEDS
*/
}

unsigned char* ledstrip::GetBuffer()
{
    return mpCurLedValues;
}

int ledstrip::GetBufferSizeinBytes()
{
    return mLedsBufSize;
}

bool ledstrip::IsBusy()
{
    return isr0Busy || isr1Busy;
}

void ledstrip::RawFill(int startLed, int numLeds, const unsigned char* pLedVals)
{
    if ((startLed < 0) || (startLed >= mLedsInStrip))
        return;
    if (numLeds >= mLedsInStrip - startLed)
        numLeds = mLedsInStrip - startLed;
    int pos = startLed * mColoursPerLed;
    unsigned char* pBuf = GetBuffer() + pos;
    memcpy(pBuf, pLedVals, numLeds * mColoursPerLed);
}

void ledstrip::HsvFill(int startLed, int numLeds, const unsigned char* pLedVals)
{
    if ((startLed < 0) || (startLed >= mLedsInStrip))
        return;
    if (numLeds >= mLedsInStrip - startLed)
        numLeds = mLedsInStrip - startLed;
    int pos = startLed * mColoursPerLed;
    unsigned char* pBuf = GetBuffer() + pos;
    
    // Copy over the values converting each to RGB
    for (int i = 0; i < numLeds; i++)
    {
        RgbColor colrVal(0,0,0);
        HsvToRgb(HsvColor(pLedVals[0],pLedVals[1] & 0xf0, (pLedVals[1] << 4) & 0xf0), colrVal);
        pBuf[pos] = colrVal.r;
        pBuf[pos+1] = colrVal.g;
        pBuf[pos+2] = colrVal.b;
//        printf("HSV %d %d %d RGB %d %d %d\r\n", pLedVals[0],pLedVals[1] & 0xf0, (pLedVals[1] << 4) & 0xf0, colrVal.r, colrVal.g, colrVal.b);
        pos += mColoursPerLed;
        pLedVals += 2;
    }
}
    
// Fill - solid colour
void ledstrip::Fill(int startLed, int numLeds, 
                int r1, int g1, int b1)
{
/*    Timer timr;
    timr.start();
*/
    if ((startLed < 0) || (startLed >= mLedsInStrip))
        return;
    if (numLeds >= mLedsInStrip - startLed)
        numLeds = mLedsInStrip - startLed;
    int pos = startLed * mColoursPerLed;
    unsigned char* pBuf = GetBuffer();
    for (int i = 0; i < numLeds; i++)
    {
        pBuf[pos] = (unsigned char) r1;
        pBuf[pos+1] = (unsigned char) g1;
        pBuf[pos+2] = (unsigned char) b1;
        pos += mColoursPerLed;
    }
/*    timr.stop();
    printf("Fill solid %d\n", timr.read_us()); // Fill 50 LEDS solid colour = 11uS
    */
}

// Fill - with interpolation of colours using HSV colour space
void ledstrip::Fill(int startLed, int numLeds, 
                int r1, int g1, int b1, 
                int r2, int g2, int b2)
{
/*    Timer timr;
    timr.start();
    */
    if ((startLed < 0) || (startLed >= mLedsInStrip))
        return;
    if (numLeds >= mLedsInStrip - startLed)
        numLeds = mLedsInStrip - startLed;
    int pos = startLed * mColoursPerLed;
    RgbColor startRGB(r1,g1,b1);
    HsvColor startHsv(0,0,0);
    RgbToHsv(startRGB, startHsv);
    RgbColor endRGB(r2,g2,b2);
    HsvColor endHsv(0,0,0);
    RgbToHsv(endRGB, endHsv);
    int curH = startHsv.h << 16;
    int curS = startHsv.s << 16;
    int curV = startHsv.v << 16;
    int interpSteps = numLeds - 1;
    if (interpSteps < 1)
        interpSteps = 1;
    int incH = ((endHsv.h - startHsv.h) << 16) / interpSteps;
    int incS = ((endHsv.s - startHsv.s) << 16) / interpSteps;
    int incV = ((endHsv.v - startHsv.v) << 16) / interpSteps;
    // Since H is a polar value we need to find out if it is best to go clockwise or anti-clockwise
    if (endHsv.h > startHsv.h)
    {
        if (endHsv.h-startHsv.h > 128)
            incH = ((startHsv.h-endHsv.h) << 16) / interpSteps;
    }
    else
    {
        // Go "round the top" using modulo result
        if (startHsv.h-endHsv.h > 128)
            incH = ((endHsv.h + 255 - startHsv.h) << 16) / interpSteps;
    }
    
//    printf("StartHSV %d %d %d EndHSV %d %d %d IncHSV %d %d %d\n", startHsv.h, startHsv.s, startHsv.v, endHsv.h, endHsv.s, endHsv.v, incH, incS, incV);
    unsigned char* pBuf = GetBuffer();
    for (int i = 0; i < numLeds; i++)
    {
        RgbColor colrVal(0,0,0);
        HsvToRgb(HsvColor((curH>>16)&0xff,curS>>16,curV>>16), colrVal);
        pBuf[pos] = colrVal.r;
        pBuf[pos+1] = colrVal.g;
        pBuf[pos+2] = colrVal.b;
//        printf("HSV %d %d %d RGB %d %d %d\n", curH>>16, curS>>16, curV>>16, colrVal.r, colrVal.g, colrVal.b);
        pos += mColoursPerLed;
        curH = curH + incH;
        curS = curS + incS;
        curV = curV + incV;
    }
    /*
    timr.stop();
    printf("Fill gradient %d\n", timr.read_us());  // Fill gradient 50 LEDS = 64uS
    */
}

void ledstrip::ShowLeds()
{
    // Check if busy
    while (isr0Busy || isr1Busy)
        wait_us(2000);
//    wait_us(2000);
    
    // Set up start points
    mCurPos0 = 0;
    mEndPos0 = mSplitPoint*mColoursPerLed;
    mCurPos1 = mSplitPoint*mColoursPerLed;
    mEndPos1 = mLedsInStrip*mColoursPerLed;
    
    // Set the buffer for the ISRs
    pLedValues0 = mpCurLedValues;
    pLedValues1 = mpCurLedValues;

    // Flip the current buffer to the alternate one for interleaved writing
    if (mpCurLedValues == mpLedValuesA)
    {
        memcpy(mpLedValuesB, mpLedValuesA, mLedsBufSize);
        mpCurLedValues = mpLedValuesB;
    }
    else
    {
        memcpy(mpLedValuesA, mpLedValuesB, mLedsBufSize);
        mpCurLedValues = mpLedValuesA;
    }
    
    // Enable interrupts
    isr0Busy = true;
    if (mSplitPoint < mLedsInStrip)
        isr1Busy = true;
    
    // Check if second strip is used
    LPC_SSP0->IMSC = (1 << SSP_IMSC_TX_RDY) & SSP_IMSC_BITMASK;
    if (mSplitPoint < mLedsInStrip && mSplitPoint > 0)
        LPC_SSP1->IMSC = (1 << SSP_IMSC_TX_RDY) & SSP_IMSC_BITMASK;
}    
