//
// Drawing Manager for LightWall
// Rob Dobson 2015
//

#include "DrawingManager.h"
#include "rtos.h"
#include "colourconverters.h"

DrawingManager::DrawingManager()
{
    pLedStrip = NULL;
    isBusy = false;
    rawFillPayloadOverhangBytes = 0;
}

void DrawingManager::Init(int numLeds, int splitPoint)
{
    pLedStrip = new ledstrip(numLeds, splitPoint);
    Thread::wait(100);
    pLedStrip->Clear();
    pLedStrip->ShowLeds();
    
}

void DrawingManager::Clear()
{
//    printf("CLEAR\r\n");
    if (pLedStrip)
        pLedStrip->Clear();
}

void DrawingManager::RawFill(char* args, unsigned char* payload, int payloadLen, int payloadOffset)
{
//    printf("RAWFILL %s payloadLen %d, payloadOffset %d\r\n", args, payloadLen, payloadOffset);
    int startLed = GetIntFromNameValPair(args, "start=", -1);
    if (startLed != -1 && payloadLen > 0)
    {
        // Include any overhang bytes from the last payload
        int numLeds = (rawFillPayloadOverhangBytes + payloadLen) / 3;
        int fromLed = startLed + (payloadOffset / 3);
        unsigned char newPayload[numLeds*3];
        memcpy(newPayload, pRawFillPayloadOverhang, rawFillPayloadOverhangBytes);
        memcpy(newPayload+rawFillPayloadOverhangBytes, payload, numLeds*3-rawFillPayloadOverhangBytes);
        
        // Send the data
//        printf("RAWFILL fromLed %d numLeds %d\r\n", fromLed, numLeds);
//        for (int i = 0; i < numLeds*3; i+=3)
//        {
//            printf("%02x %02x %02x\r\n", newPayload[i], newPayload[i+1], newPayload[i+2]);
//        }
        pLedStrip->RawFill(fromLed, numLeds, newPayload);

        // Save any overhanging bytes for the next fill
        int overhangStart = (numLeds * 3) - rawFillPayloadOverhangBytes;
        rawFillPayloadOverhangBytes = rawFillPayloadOverhangBytes + payloadLen - (numLeds * 3);
        for (int i = 0; i < rawFillPayloadOverhangBytes; i++)
        {
            pRawFillPayloadOverhang[i] = payload[overhangStart + i];
        }
    }
}

void DrawingManager::Fill(char* args)
{
//    printf("FILL %s\r\n", args);
    int startLed = GetIntFromNameValPair(args, "start=", -1);
    int numLeds = GetIntFromNameValPair(args, "len=", -1);
    int r1 = GetIntFromNameValPair(args, "r1=", -1);
    int g1 = GetIntFromNameValPair(args, "g1=", -1);
    int b1 = GetIntFromNameValPair(args, "b1=", -1);
    int r2 = GetIntFromNameValPair(args, "r2=", -1);
    int g2 = GetIntFromNameValPair(args, "g2=", -1);
    int b2 = GetIntFromNameValPair(args, "b2=", -1);
    if (startLed != -1 && numLeds != -1 && r1 != -1 && g1 != -1 && b1 != -1)
    {
        if (r2 == -1 || g2 == -1 || b2 == -1)
            pLedStrip->Fill(startLed, numLeds, r1, g1, b1);
        else
            pLedStrip->Fill(startLed, numLeds, r1, g1, b1, r2, g2, b2);
    }
}

void DrawingManager::ShowLeds()
{
//    printf("SHOWLEDS\r\n");
    rawFillPayloadOverhangBytes = 0;
    if (pLedStrip)
        pLedStrip->ShowLeds();
}

void DrawingManager::DisplayIdle(unsigned int stepCount)
{
    // Display a step in an auto sequence
    if (!pLedStrip)
        return;
    if (pLedStrip->IsBusy())
        return;
    pLedStrip->Clear();
    int numSnakes = 15;
    int ledsPerGroup = pLedStrip->GetNumLeds() / numSnakes;
    int snakeLen = 10;
    int snakeStep = stepCount % ledsPerGroup;
    int colrBase = (stepCount / ledsPerGroup);
    // Create a set of colourful snakes that roam around the wall
    for (int i = 0; i < numSnakes; i ++)
    {
        HsvColor hsv1(((colrBase + i) * 237) % 255, 128, 10);
        RgbColor rgb1(0,0,0);
        HsvToRgb(hsv1, rgb1);
        HsvColor hsv2(((colrBase + i + 27) * 13) % 255, 255, 255);
        RgbColor rgb2(0,0,0);
        HsvToRgb(hsv2, rgb2);
        pLedStrip->Fill((i*ledsPerGroup)+snakeStep,snakeLen/2, rgb1.r, rgb1.g, rgb1.b, rgb2.r, rgb2.g, rgb2.b);
        pLedStrip->Fill((i*ledsPerGroup)+snakeStep+snakeLen/2, snakeLen/2, rgb2.r, rgb2.g, rgb2.b, rgb1.r, rgb1.g, rgb1.b);
    }
    pLedStrip->ShowLeds();
}

int DrawingManager::GetIntFromNameValPair(char* buf, char* name, int invalidVal)
{
    int val = invalidVal;
    char* pFnd = strstr(buf, name);
    if (pFnd)
        val = atoi(pFnd + strlen(name));
    return val;
}

