// RdWebServer - Simple Web Server for MBED
// Copyright (C) Rob Dobson 2013-2015, MIT License
// Inspired by Jasper Schuurmans multi-threaded web server for Netduino which now seems to have gone from ...
// http://www.schuurmans.cc/multi-threaded-web-server-for-netduino-plus
// More details at http://robdobson.com/2013/10/moving-my-window-shades-control-to-mbed/

#include "RdWebServerDefs.h"
#include "RdWebServer.h"

// Limits - note particularly the MAX_FILENAME_LEN
const int MAX_FILENAME_LEN = (MAX_ARGSTR_LEN + 20);

// Constructor
RdWebServer::RdWebServer(TCPSocketServer& tcpServerSocket, Mutex* pSdCardMutex)
    : _serverSocket(tcpServerSocket)
{
    _initOk = false;
    _pStatusLed = NULL;
    _curSplitPayloadPos = -1;
    _curCmdStr[0] = 0;
    _curArgStr[0] = 0;
    _bufferReceivedLen = 0;
    _curContentLen = -1;
    _curHttpMethod = METHOD_OTHER;
    _curWebServerCmdDef = NULL;
    _pSdCardMutex = pSdCardMutex;
    _numWebServerCmds = 0;
    _numWebServerCachedFiles = 0;
}

// Destructor
RdWebServer::~RdWebServer()
{
    // Clean-up - probably never called as we're on a microcontroller!
    for (int i = 0; i < _numWebServerCmds; i++)
        delete _pWebServerCmds[i];
    _numWebServerCmds = 0;
}

// Init
bool RdWebServer::init(int port, DigitalOut* pStatusLed, char* pBaseWebFolder)
{
    // Settings
    _port = port;
    _pStatusLed = pStatusLed;
    _pBaseWebFolder = pBaseWebFolder;

    // Non-blocking by default
    _blockingOnAccept = false;
    _blockingOnReceive = false;
    
    // This is the same as the default in the socket.cpp file
    _timeoutOnBlocking = 1500;
    
    // If closeConnAfterSend is set false then the socket connection remains open and only
    // one client can access the server until a time-out occurs - this should be ok for most applications
    _closeConnAfterSend = false;
    _closeConnOnReceiveFail = true;

    if(_serverSocket.bind(port)< 0) 
    {
        RD_WARN("TCP server bind fail");
        return false;
    }
    
    // Only one connection at a time as the server is single-threaded
    if(_serverSocket.listen(1) < 0)
    {
        RD_WARN("TCP server listen fail");
        return false;
    }
    RD_INFO("TCP server is listening...");
    _initOk = true;
    return true;
}

// Run - never returns so run in a thread
void RdWebServer::run()
{
    // Check initialised ok
    if (!_initOk)
        return;

    // Start accepting connections
    RD_INFO("Waiting for TCP connection");
    while (true)
    {
        // Accept connection if available
        TCPSocketConnection clientSocketConn;
        clientSocketConn.set_blocking(_blockingOnAccept, _timeoutOnBlocking);
        _serverSocket.accept(clientSocketConn);
        if (!clientSocketConn.is_connected())
            continue;
        
        // Connection
        RD_INFO("Connection from IP: %s", clientSocketConn.get_address());
        if (_pStatusLed != NULL)
            *_pStatusLed = true;
        
        // While connected
        bool forcedClosed = false;
        _curSplitPayloadPos = -1;
        while(clientSocketConn.is_connected() && !forcedClosed)
        {
            // Receive data
            if (_blockingOnReceive != _blockingOnAccept)
                clientSocketConn.set_blocking(_blockingOnReceive, _timeoutOnBlocking);
            int rxLen = clientSocketConn.receive(_buffer, HTTPD_MAX_REQ_LENGTH);
            if (rxLen == -1)
            {
                if (_closeConnOnReceiveFail)
                {
                    int closeRet = clientSocketConn.close();
                    RD_DBG("Rx Timeout - conn close() ret %d (is_connected %d)", closeRet, clientSocketConn.is_connected());
                    forcedClosed = true;
                }
                else
                {
                    RD_DBG("Rx Timeout - not closing - is_connected %d", clientSocketConn.is_connected());
                }
                continue;
            }
            if (rxLen == 0)
            {
                RD_DBG("clientSocketConn.receive() returned %d - ignoring -  is connected %d", rxLen, clientSocketConn.is_connected());
                continue;
            }
            
            // Handle received message
            _buffer[rxLen] = '\0';
            _bufferReceivedLen = rxLen;
            handleReceivedHttp(clientSocketConn);
            
            // Close connection if required
            if (_closeConnAfterSend)
            {
                int closeRet = clientSocketConn.close();
                RD_DBG("After send connection close() ret %d is connected %d", closeRet, clientSocketConn.is_connected());
                forcedClosed = true;
            }
        }
        
        // Waiting again ...
        RD_INFO("Waiting for TCP connection");
    }
}

// Handle a request            
bool RdWebServer::handleReceivedHttp(TCPSocketConnection &clientSocketConn)
{
    bool handledOk = false;
    
    // Check for split payload
    if (_curSplitPayloadPos != -1)
    {
        // Handle remaining parts of content
        char* respStr = (_curWebServerCmdDef->_callback)(_curHttpMethod, _curCmdStr, _curArgStr, _buffer, _bufferReceivedLen, 
                        _curContentLen, (unsigned char*)_buffer, _bufferReceivedLen, _curSplitPayloadPos);
        RD_DBG("Received part of message - content %d - rx %d - splitAfter %d", _curContentLen, _bufferReceivedLen, _curSplitPayloadPos);
        _curSplitPayloadPos += _bufferReceivedLen;
        // Check if all received
        if (_curSplitPayloadPos >= _curContentLen)
        {
            _curSplitPayloadPos = -1;
            clientSocketConn.send_all(respStr, strlen(respStr));
            RD_DBG("That was the end of message - content %d - rx %d", _curContentLen, _bufferReceivedLen);
        }
        return true;
    }

    // Get payload information    
    int payloadLen = -1;
    unsigned char* pPayload = getPayloadDataFromMsg(_buffer, _bufferReceivedLen, payloadLen);
    
    // Debug
    int displayLen = _bufferReceivedLen;
    if (payloadLen > 0)
        displayLen = _bufferReceivedLen-payloadLen;
    RD_DBG("\r\nNEW REQUEST RxLen %d ContentLen %d PayloadLen %d\n\r\n\r%.*s", _bufferReceivedLen, _curContentLen, payloadLen, displayLen, _buffer);
    
    // Get HTTP method
    _curHttpMethod = METHOD_OTHER;
    if (strncmp(_buffer, "GET ", 4) == 0)
        _curHttpMethod = METHOD_GET;
    else if (strncmp(_buffer, "POST", 4) == 0)
        _curHttpMethod = METHOD_POST;
    else if (strncmp(_buffer, "OPTIONS", 7) == 0)
        _curHttpMethod = METHOD_OPTIONS;

    // See if there is a valid HTTP command
    _curContentLen = -1;
    if (extractCmdArgs(_buffer+3, _curCmdStr, MAX_CMDSTR_LEN, _curArgStr, MAX_ARGSTR_LEN, _curContentLen))
    {
        RD_DBG("CmdStr %s", _curCmdStr);
        RD_DBG("ArgStr %s", _curArgStr);

        bool cmdFound = false;
        for (int wsCmdIdx = 0; wsCmdIdx < _numWebServerCmds; wsCmdIdx++)
        {
            RdWebServerCmdDef* pCmd = _pWebServerCmds[wsCmdIdx];
            if (strcasecmp(pCmd->_pCmdStr, _curCmdStr) == 0)
            {
                RD_DBG("FoundCmd <%s> Type %d", _curCmdStr, pCmd->_cmdType);
                cmdFound = true;
                if (pCmd->_cmdType == RdWebServerCmdDef::CMD_CALLBACK)
                {
                    char* respStr = (pCmd->_callback)(_curHttpMethod, _curCmdStr, _curArgStr, _buffer, _bufferReceivedLen, 
                                    _curContentLen, pPayload, payloadLen, 0);
                    // Handle split-payload situation
                    if (_curContentLen > 0 && payloadLen < _curContentLen)
                    {
                        _curSplitPayloadPos = payloadLen;
                        _curWebServerCmdDef = pCmd;
                    }
                    else
                    {
                        clientSocketConn.send_all(respStr, strlen(respStr));
                    }
                    handledOk = true;
                }
                else if ( (pCmd->_cmdType == RdWebServerCmdDef::CMD_LOCALFILE) ||
                          (pCmd->_cmdType == RdWebServerCmdDef::CMD_SDORUSBFILE) )
                {
                    char combinedFileName[MAX_FILENAME_LEN];
                    strcpy(combinedFileName, pCmd->_substFileName);
                    if (strlen(combinedFileName) == 0)
                        strcpy(combinedFileName, _curCmdStr);
                    else if (combinedFileName[strlen(combinedFileName)-1] == '*')
                        strcpy(combinedFileName+strlen(combinedFileName)-1, _curArgStr);
                    if (pCmd->_cmdType == RdWebServerCmdDef::CMD_LOCALFILE)
                        handledOk = handleLocalFileRequest(combinedFileName, _curArgStr, clientSocketConn, pCmd->_bCacheIfPossible);
                    else
                        handledOk = handleSDFileRequest(combinedFileName, _curArgStr, clientSocketConn);

                }
                break;
            }
        }
        // If command not found see if it is a local file
        if (!cmdFound)
        {
            char combinedFileName[MAX_FILENAME_LEN];
            strcpy(combinedFileName, _curCmdStr);
            if (strlen(_curArgStr) > 0)
            {
                strcat(combinedFileName, "/");
                strcat(combinedFileName, _curArgStr);
            }
            handledOk = handleSDFileRequest(combinedFileName, _curArgStr, clientSocketConn);
        }
    }
    else
    {
        RD_DBG("Cannot find command or args\r\n");
    }
    return handledOk;
}

// Add a command to the server
void RdWebServer::addCommand(char* pCmdStr, int cmdType, CmdCallbackType callback, char* substFileName, bool cacheIfPossible)
{
    // Check for overflow
    if (_numWebServerCmds >= MAX_WEB_SERVER_CMDS)
        return;
        
    // Create new command definition and add
    RdWebServerCmdDef* pNewCmdDef = new RdWebServerCmdDef(pCmdStr, cmdType, callback, substFileName, cacheIfPossible);
    _pWebServerCmds[_numWebServerCmds] = pNewCmdDef;
    _numWebServerCmds++;
}

// Form a header to respond
void RdWebServer::formHTTPHeader(const char* rsltCode, const char* contentType, int contentLen)
{
    const char* closeConnStr = "\r\nConnection: keep-alive";
    if(contentLen != -1)
        sprintf(_httpHeader, "HTTP/1.1 %s\r\nAccess-Control-Allow-Origin: *\r\nContent-Type: %s\r\nContent-Length: %d%s\r\n\r\n", rsltCode, contentType, contentLen, _closeConnAfterSend ? closeConnStr : "");
    else
        sprintf(_httpHeader, "HTTP/1.1 %s\r\nAccess-Control-Allow-Origin: *\r\nContent-Type: %s%s\r\n\r\n", rsltCode, contentType, _closeConnAfterSend ? closeConnStr : "");
}

// Use file extension to determine MIME type
char* getMimeTypeStr(char* filename)
{
    char* mimeType = "text/plain";
    char *pDot = strrchr(filename, '.');
    if (pDot && (!strcmp(pDot, ".html") || !strcmp(pDot, ".htm")))
        mimeType = "text/html";
    else if (pDot && !strcmp(pDot, ".css"))
        mimeType = "text/css";
    else if (pDot && !strcmp(pDot, ".js"))
        mimeType = "application/javascript";
    else if (pDot && !strcmp(pDot, ".png"))
        mimeType = "image/png";
    else if (pDot && (!strcmp(pDot, ".jpeg") || !strcmp(pDot, ".jpeg")))
        mimeType = "image/jpeg";
    return mimeType;
}

// Handle request for files/listing from SDCard
bool RdWebServer::handleSDFileRequest(char* inFileName, char* argStr, TCPSocketConnection &clientSocketConn)
{
#ifdef SUPPORT_SD_FILESYSTEM

    bool handledOk = false;
    const int HTTPD_MAX_FNAME_LENGTH = 127;
    char filename[HTTPD_MAX_FNAME_LENGTH+1];
    
    RD_INFO("Requesting file %s", inFileName);
    
#ifdef SUPPORT_FOLDER_VIEW
    if ((strlen(inFileName) > 0) && (inFileName[strlen(inFileName)-1] == '/'))
    {
        RD_INFO("Request directory %s%s", _pBaseWebFolder, inFileName);
        sprintf(filename, "%s%s", _pBaseWebFolder, inFileName);
        
        // Obtain lock on SD card (if reqd)
        if (_pSdCardMutex)
            _pSdCardMutex->lock();
            
        DIR *d = opendir(filename);
        if (d != NULL) 
        {
            formHTTPHeader("200 OK", "text/html", -1);
            clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
            sprintf(_httpHeader,"<html><head><title>Directory Listing</title></head><body><h1>%s</h1><ul>", inFileName);
            clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
            struct dirent *p;
            while((p = readdir(d)) != NULL) 
            {
                RD_INFO("%s", p->d_name);
                sprintf(_httpHeader,"<li>%s</li>", p->d_name);
                clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
            }
        }
        closedir(d);
        
        // Release lock on SD card (if reqd)
        if (_pSdCardMutex)
            _pSdCardMutex->unlock();

        RD_DBG("Directory closed\n");
        sprintf(_httpHeader,"</ul></body></html>");
        clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
        handledOk = true;
    }
    else
#endif
    {
        sprintf(filename, "%s%s", _pBaseWebFolder, inFileName);
        RD_INFO("Filename %s", filename);
            
        // Obtain lock on SD card (if reqd)
        if (_pSdCardMutex)
            _pSdCardMutex->lock();

        FILE* fp = fopen(filename, "r");
        if (fp == NULL)
        {
            RD_WARN("Filename %s not found", filename);
            formHTTPHeader("404 Not Found", "text/plain", 0);
            clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
            handledOk = true;
        }
        else
        {
            RD_INFO("Sending file %s", filename);
            // Find file length
            fseek(fp, 0L, SEEK_END);
            int sz = ftell(fp);
            fseek(fp, 0L, SEEK_SET);
            // Get mime type
            char* mimeType = getMimeTypeStr(filename);
            RD_DBG("MIME TYPE %s", mimeType);
            // Form header   
            formHTTPHeader("200 OK", mimeType, sz);
            clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
            // Read file in blocks and send
            int rdCnt = 0;
            char fileBuf[1024];
            while ((rdCnt = fread(fileBuf, sizeof( char ), 1024, fp)) == 1024) 
            {
                clientSocketConn.send_all(fileBuf, rdCnt);
            }
            clientSocketConn.send_all(fileBuf, rdCnt);
            fclose(fp);
            handledOk = true;
        }
        
        // Obtain lock on SD card (if reqd)
        if (_pSdCardMutex)
            _pSdCardMutex->unlock();

    }
    return handledOk;
    
#else // support SUPPORT_SD_FILESYSTEM

    return false;
    
#endif
}

// Send a file from cache - used for local files only
#ifdef SUPPORT_LOCAL_FILE_CACHE
void RdWebServer::sendFromCache(RdFileCacheEntry* pCacheEntry, TCPSocketConnection &clientSocketConn)
{
    RD_INFO("Sending file %s from cache %d bytes", pCacheEntry->_fileName, pCacheEntry->_nFileLen);
    // Get mime type
    char* mimeType = getMimeTypeStr(pCacheEntry->_fileName);
    RD_INFO("MIME TYPE %s", mimeType);
    // Form header
    formHTTPHeader("200 OK", mimeType, pCacheEntry->_nFileLen);
    clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
    char* pMem = pCacheEntry->_pFileContent;
    int nLenLeft = pCacheEntry->_nFileLen;
    int blkSize = 2048;
    while(nLenLeft > 0)
    {
        if (blkSize > nLenLeft)
            blkSize = nLenLeft;
        clientSocketConn.send_all(pMem, blkSize);
        pMem += blkSize;
        nLenLeft -= blkSize;
    }
}
#endif

// Handle a request for a local file
bool RdWebServer::handleLocalFileRequest(char* inFileName, char* argStr, TCPSocketConnection &clientSocketConn, bool bCacheIfPossible)
{

#ifdef SUPPORT_LOCAL_FILESYSTEM

    const int HTTPD_MAX_FNAME_LENGTH = 127;
    char localFilename[HTTPD_MAX_FNAME_LENGTH+1];
    char reqFileNameStr[HTTPD_MAX_FNAME_LENGTH+1];

    RD_INFO("Requesting local file %s", inFileName);
    sprintf(reqFileNameStr, "/%s", inFileName);
    sprintf(localFilename, "/local/%s", inFileName);
        
    if (bCacheIfPossible)
    {
        // Check if file already cached
        bool bTryToCache = true;
        for (int wsCacheIdx = 0; wsCacheIdx < _numWebServerCachedFiles; wsCacheIdx++)
        {
            RdFileCacheEntry* pCachedFile = _pWebServerCachedFiles[wsCacheIdx];
            if (strcmp(inFileName, pCachedFile->_fileName) == 0)
            {
                if (pCachedFile->_bCacheValid)
                {
                    sendFromCache(pCachedFile, clientSocketConn);
                    return true;
                }
                bTryToCache = false; // don't try to cache as cacheing must have already failed
            }
        }
        
        // Check for cache full
        if (_numWebServerCachedFiles >= MAX_WEB_SERVER_CACHED_FILES)
            bTryToCache = false;
            
        // See if we can cache the file
        if (bTryToCache)
        {
            RdFileCacheEntry* pCacheEntry = new RdFileCacheEntry(inFileName);
            if (pCacheEntry)
            {
                bool bCacheSuccess = pCacheEntry->readLocalFileIntoCache(localFilename);

                // Store the cache entry even if reading failed (mem alloc or file not found, etc) so we don't try to cache again
                _pWebServerCachedFiles[_numWebServerCachedFiles] = pCacheEntry;
                _numWebServerCachedFiles++;
                if (bCacheSuccess)
                {
                    sendFromCache(pCacheEntry, clientSocketConn);
                    return true;
                }
            }
        }        
    }
    
    LocalFileSystem local("local");
    
    FILE* fp = fopen(localFilename, "r");
    if (fp == NULL)
    {
        RD_WARN("Local file %s not found", localFilename);
        formHTTPHeader("404 Not Found", "text/plain", 0);
        clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
        return true;
    }
    else
    {
        RD_INFO("Sending file %s from disk", localFilename);
        // Find file length
        fseek(fp, 0L, SEEK_END);
        int sz = ftell(fp);
        fseek(fp, 0L, SEEK_SET);
        // Get mime type
        char* mimeType = getMimeTypeStr(localFilename);
        RD_DBG("MIME TYPE %s", mimeType);
        // Form header   
        formHTTPHeader("200 OK", mimeType, sz);
        clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
        int rdCnt = 0;
        while ((rdCnt = fread(_httpHeader, sizeof( char ), sizeof(_httpHeader), fp)) == sizeof(_httpHeader)) 
        {
            clientSocketConn.send_all(_httpHeader, rdCnt);
        }
        if (rdCnt != 0)
            clientSocketConn.send_all(_httpHeader, rdCnt);
        fclose(fp);
        return true;
    }
    
#else

    RD_WARN("Local file system not supported");
    formHTTPHeader("404 Not Found", "text/plain", 0);
    clientSocketConn.send_all(_httpHeader,strlen(_httpHeader));
    return true;

#endif
}

// Read file into cache
#ifdef SUPPORT_LOCAL_FILE_CACHE
bool RdFileCacheEntry::readLocalFileIntoCache(char* fileName)
{
#ifdef SUPPORT_LOCAL_FILESYSTEM
    
    RD_INFO("Reading into cache %s", fileName);
    LocalFileSystem local("local");
    FILE* fp = fopen(fileName, "r");
    if (fp == NULL)
    {
        RD_WARN("Failed to open file");
        return false;
    }
    RD_DBG("Seeking ");
    fseek(fp, 0, SEEK_END);
    _nFileLen = (int)ftell(fp);
    RD_DBG("Len %d ", _nFileLen);
    _pFileContent = new char[_nFileLen];
    RD_DBG("Buf %08x ", _pFileContent);
    if (!_pFileContent)
    {
        RD_WARN("Failed to allocate %lu", _nFileLen);
        fclose(fp);
        return false;
    }
    RD_DBG("Allocated");
    memset(_pFileContent, 0, _nFileLen);
    fseek(fp, 0, SEEK_SET);
    char* pMem = _pFileContent;
    char fileBuf[100];
    int totCnt = 0;
    int rdCnt = 0;
    RD_DBG("Reading");
    while (true)
    {
        int toRead = _nFileLen - totCnt;
        if (toRead > sizeof(fileBuf))
            toRead = sizeof(fileBuf);
        if (toRead <= 0)
            break;
        rdCnt = fread(fileBuf, sizeof(char), toRead, fp);
        if (rdCnt <= 0)
            break;
        RD_DBG("Read %d tot %d of %d", rdCnt, totCnt, _nFileLen);
        memcpy(pMem, fileBuf, rdCnt);
        pMem += rdCnt;
        totCnt += rdCnt;
    }
    RD_DBG("Done read");
    fclose(fp);
    RD_DBG("Success in caching %d bytes (read %d)", _nFileLen, totCnt);
    _bCacheValid = true;
    return true;
#else
    return false;
#endif
}

#endif

// Extract arguments from command
bool RdWebServer::extractCmdArgs(char* buf, char* pCmdStr, int maxCmdStrLen, char* pArgStr, int maxArgStrLen, int& contentLen)
{
    contentLen = -1;
    *pCmdStr = '\0';
    *pArgStr = '\0';
    int cmdStrLen = 0;
    int argStrLen = 0;
    if (buf == NULL)
        return false;
    // Check for Content-length header
    char* contentLenText = "Content-Length:";
    char* pContLen = strstr(buf, contentLenText);
    if (pContLen)
    {
        if (*(pContLen + strlen(contentLenText)) != '\0')
            contentLen = atoi(pContLen + strlen(contentLenText));
    }
    
    // Check for first slash
    char* pSlash1 = strchr(buf, '/');
    if (pSlash1 == NULL)
        return false;
    pSlash1++;
    // Extract command
    while(*pSlash1)
    {
        if (cmdStrLen >= maxCmdStrLen-1)
            break;
        if ((*pSlash1 == '/') || (*pSlash1 == ' ') || (*pSlash1 == '\n') || (*pSlash1 == '?') || (*pSlash1 == '&'))
            break;
        *pCmdStr++ = *pSlash1++;
        *pCmdStr = '\0';
        cmdStrLen++;
    }
    if ((*pSlash1 == '\0') || (*pSlash1 == ' ') || (*pSlash1 == '\n'))
        return true;
    // Now args
    *pSlash1++;
    while(*pSlash1)
    {
        if (argStrLen >= maxArgStrLen-1)
            break;
        if ((*pSlash1 == ' ') || (*pSlash1 == '\n'))
            break;
        *pArgStr++ = *pSlash1++;
        *pArgStr = '\0';
        argStrLen++;
    }
    return true;
}

unsigned char* RdWebServer::getPayloadDataFromMsg(char* msgBuf, int msgLen, int& payloadLen)
{
    payloadLen = -1;
    char* ptr = strstr(msgBuf, "\r\n\r\n");
    if (ptr)
    {
        payloadLen = msgLen - (ptr+4-msgBuf);
        return (unsigned char*) (ptr+4);
    }
    return NULL;
}

int RdWebServer::getContentLengthFromMsg(char* msgBuf)
{
    char* ptr = strstr(msgBuf, "Content-Length:");
    if (ptr)
    {
        ptr += 15;
        int contentLen = atoi(ptr);
        if (contentLen >= 0)
            return contentLen;
    }
    return 0;
}
