// RdWebServer - Simple Web Server for MBED
// Copyright (C) Rob Dobson 2013-2015, MIT License
// Inspired by Jasper Schuurmans multi-threaded web server for Netduino which now seems to have gone from ...
// http://www.schuurmans.cc/multi-threaded-web-server-for-netduino-plus
// More details at http://robdobson.com/2013/10/moving-my-window-shades-control-to-mbed/

#ifndef RD_WEB_SERVER
#define RD_WEB_SERVER

#include "mbed.h"

// Debug level
#ifdef RDWEB_DEBUG 
#if (RDWEB_DEBUG > 3)
#define RD_DBG(x, ...) std::printf("[RD_DBG: %s:%d]" x "\r\n", __FILE__, __LINE__, ##__VA_ARGS__);
#else
#define RD_DBG(x, ...)
#endif
#else
#define RD_DBG(x, ...)
#endif

#ifdef RDWEB_DEBUG 
#if (RDWEB_DEBUG > 2)
#define RD_INFO(x, ...) std::printf("[RD_INFO: %s:%d]" x "\r\n", __FILE__, __LINE__, ##__VA_ARGS__);
#else
#define RD_INFO(x, ...)
#endif
#else
#define RD_INFO(x, ...)
#endif

#ifdef RDWEB_DEBUG 
#if (RDWEB_DEBUG > 1)
#define RD_WARN(x, ...) std::printf("[RD_WARNING: %s:%d]" x "\r\n", __FILE__, __LINE__, ##__VA_ARGS__);
#else
#define RD_WARN(x, ...)
#endif
#else
#define RD_WARN(x, ...)
#endif

#ifdef RDWEB_DEBUG 
#if (RDWEB_DEBUG > 0)
#define RD_ERR(x, ...) std::printf("[RD_ERR: %s:%d]" x "\r\n", __FILE__, __LINE__, ##__VA_ARGS__);
#else
#define RD_ERR(x, ...)
#endif
#else
#define RD_ERR(x, ...)
#endif

// Length of strings
const int MAX_CMDSTR_LEN = 100;
const int MAX_ARGSTR_LEN = 100;
const int MAX_WEB_SERVER_CMDS = 30;
const int MAX_WEB_SERVER_CACHED_FILES = 5;
const int SUBST_MAX_FNAME_LEN = 40;  // note local files on MBED are 8.3 but this might include other files

extern RawSerial pc;

class RdFileCacheEntry
{
    public:
        RdFileCacheEntry(char* pFileName)
        {
            _bCacheValid = false;
            strncpy(_fileName, pFileName, SUBST_MAX_FNAME_LEN-1);
            _fileName[SUBST_MAX_FNAME_LEN-1] = '\0';
            _pFileContent = NULL;
        }
        ~RdFileCacheEntry()
        {
            delete _pFileContent;
        }
        bool readLocalFileIntoCache(char* fileName);
    public:
        bool _bCacheValid;
        char _fileName[SUBST_MAX_FNAME_LEN];
        char* _pFileContent;
        int _nFileLen;
};

typedef char* (*CmdCallbackType)(int method, char*cmdStr, char* argStr, char* msgBuffer, int msgLen, 
                int contentLen, unsigned char* pPayload, int payloadLen, int splitPayloadPos);

class RdWebServerCmdDef
{
    public:
        static const int CMD_LOCALFILE = 1;
        static const int CMD_CALLBACK = 2;
        static const int CMD_SDORUSBFILE = 3;
        RdWebServerCmdDef(char* pStr, int cmdType, CmdCallbackType callback, char* substFileName = NULL, bool bCacheIfPossible = false)
        {
            _pCmdStr = pStr;
            _cmdType = cmdType;
            _callback = callback;
            _substFileName[0] = '\0';
            if (substFileName != NULL)
            {
                strncpy(_substFileName, substFileName, SUBST_MAX_FNAME_LEN);
                _substFileName[SUBST_MAX_FNAME_LEN-1] = '\0';
            }
            _bCacheIfPossible = bCacheIfPossible;
        };
        char* _pCmdStr;
        int _cmdType;
        CmdCallbackType _callback;
        char _substFileName[SUBST_MAX_FNAME_LEN];
        bool _bCacheIfPossible;
};

const int HTTPD_MAX_HDR_LENGTH = 255;
// The FRDM-K64F has more memory than the LPC1768 so allow a larger buffer
#if defined(MCU_MK64F12)
const int HTTPD_MAX_REQ_LENGTH = 2048;
#warning("TCP Request Length 2048")
#elif defined(TARGET_ARCH_BLE)
const int HTTPD_MAX_REQ_LENGTH = 1024;
#warning("TCP Request Length 1024")
#else
const int HTTPD_MAX_REQ_LENGTH = 1024;
#warning("TCP Request Length 1024")
#endif

class TCPSocketServer;
class TCPSocketConnection;
class Mutex;

class RdWebServer
{
    public :
        static const int METHOD_OTHER = 0;
        static const int METHOD_GET = 1;
        static const int METHOD_POST = 2;
        static const int METHOD_OPTIONS = 3;
        RdWebServer(TCPSocketServer& tcpServerSocket, Mutex* pSdCardMutex = NULL);
        virtual ~RdWebServer();
        
        bool init(int port, DigitalOut* pStatusLed, char* pBaseWebFolder);
        void run();
        void addCommand(char* pCmdStr, int cmdType, CmdCallbackType callback = NULL, char* substFileName = NULL, bool cacheIfPossible = false);
        
        static unsigned char* getPayloadDataFromMsg(char* msgBuf, int msgLen, int& payloadLen);
        static int getContentLengthFromMsg(char* msgBuf);

    private :
        int _port;
        DigitalOut* _pStatusLed;
        TCPSocketServer& _serverSocket;
        bool _initOk;
        RdWebServerCmdDef* _pWebServerCmds[MAX_WEB_SERVER_CMDS];
        int _numWebServerCmds;
        RdFileCacheEntry* _pWebServerCachedFiles[MAX_WEB_SERVER_CACHED_FILES];
        int _numWebServerCachedFiles;
        bool extractCmdArgs(char* buf, char* pCmdStr, int maxCmdStrLen, char* pArgStr, int maxArgStrLen, int& contentLen);
        char* _pBaseWebFolder;
        char _httpHeader[HTTPD_MAX_HDR_LENGTH+1];
        char _buffer[HTTPD_MAX_REQ_LENGTH+1];
        int _bufferReceivedLen;

        bool handleLocalFileRequest(char* cmdStr, char* argStr, TCPSocketConnection &client, bool bCacheIfPossible);
        bool handleSDFileRequest(char* cmdStr, char* argStr, TCPSocketConnection &client);
        void handleCGIRequest(char* pUriStr, char* pQueryStr);
        void sendFromCache(RdFileCacheEntry* pCacheEntry, TCPSocketConnection &client);
        bool handleReceivedHttp(TCPSocketConnection &client);
        void formHTTPHeader(const char* rsltCode, const char* contentType, int contentLen);

        // Settings - see the constructor
        bool _blockingOnAccept;
        bool _blockingOnReceive;
        int _timeoutOnBlocking;
        bool _closeConnAfterSend;
        bool _closeConnOnReceiveFail;
        
        // Handling of split payloads on receipt (e.g. POST)
        int _curSplitPayloadPos;
        char _curCmdStr[MAX_CMDSTR_LEN];
        char _curArgStr[MAX_ARGSTR_LEN];
        int _curContentLen;
        int _curHttpMethod;
        RdWebServerCmdDef* _curWebServerCmdDef;
        
        // Mutex for SD card access (if supplied in constructor)
        Mutex* _pSdCardMutex;
};

#endif
