// Gas usage based on pulses from a gas meter
// Rob Dobson 2015

#include "GasUseCounter.h"

// Callback from web server to handle getting current gas count
char* GasUseCounter::getGasUseCallback(char* cmdStr, char* argStr)
{
    sprintf(_gasCountStr, "{\"gasUseCount\":\"%d\"}", GetCount());
    return _gasCountStr;
}

// Init
void GasUseCounter::Init()
{
    GetGasCountFromSD();
}
    
// Service function
bool GasUseCounter::Service()
{
    // Check for an edge
    bool edgeDetected = _pulseDetector->Service();
    if (edgeDetected)
    {
        // Check if we need to store in non-volatile storage
        if (GetCount() >= _lastWrittenGasCount + MAX_PULSES_BEFORE_STORE_NV)
        {
            WriteGasCountToSD();
            _lastWrittenGasCount = GetCount();
        }
            
        // Show count
        _logger.LogDebug("GasCount %d Rate(ms) %d", _pulseDetector->GetPulseCount(), _pulseDetector->GetPulseRateMs());
    }   
    return edgeDetected;
}
    
// Get the current usage count from SD card
void GasUseCounter::GetGasCountFromSD()
{
    int count0 = 0;
    int count1 = 0;
    for (int i = 0; i < 2; i++)
    {
        const char* fname = _gasUseFilename1;
        if (i == 1)
            fname = _gasUseFilename2;

        // Obtain lock to access sd card
        _sdCardMutex.lock();
        
        FILE* fp = fopen(fname, "r");
        if (fp == NULL)
        {
            // Release lock on sd card
            _sdCardMutex.unlock();

            _logger.LogDebug("GasCount Read Filename %s not found", fname);            
        }
        else
        {
            int curCount = 0;
            int retVal = fscanf(fp, "%d", &curCount);
            fclose(fp);
        
            // Release lock on sd card
            _sdCardMutex.unlock();

            // Handle result
            if (retVal == 1)
                _logger.LogDebug("GasCount read from file %s gas count = %d", fname, curCount);
            else
                _logger.LogDebug("GasCount Failed to read gas count from file %s", fname);
            if (i == 0)
                count0 = curCount;
            else
                count1 = curCount;
        }
    }
    // Find the highest count
    int maxCount = count0;
    if (count1 > count0)
        maxCount = count1;
    _lastWrittenGasCount = maxCount;
    _pulseDetector->SetPulseCount(maxCount);
    _logger.LogDebug("GasCount set to %d", maxCount);
}

void GasUseCounter::WriteGasCountToSD()
{
    for (int i = 0; i < 2; i++)
    {
        const char* fname = _gasUseFilename1;
        if (i == 1)
            fname = _gasUseFilename2;

        // Obtain lock to access sd card
        _sdCardMutex.lock();

        FILE* fp = fopen(fname, "w");
        if (fp == NULL)
        {
            _logger.LogDebug("GasCount write failed filename %s not found", fname);
        }
        else
        {
            fprintf(fp, "%d", GetCount());
            _logger.LogDebug("GasCount written to %s gas count = %d", fname, GetCount());
            fclose(fp);
        }
        
        // Release lock on sd card
        _sdCardMutex.unlock();
        
    }
}
