// Log to SD
// Rob Dobson, 2015

#include "Logger.h"

Logger::Logger(const char* eventLogFileName, const char* dataLogFileBase, Mutex &sdCardMutex) :
            _sdCardMutex(sdCardMutex)
{
    _eventLogFileName = eventLogFileName;
    _dataLogFileBase = dataLogFileBase;
    _logDebugToFile = false;
    _logDebugToConsole = false;
}

void Logger::LogEvent(const char* format, ...)
{
    char timeBuf[40];
    time_t seconds = time(NULL);
    strftime(timeBuf, sizeof(timeBuf), "%Y-%m-%d\t%H:%M:%S\t", localtime(&seconds));

    // Obtain lock to access sd card - if unable then abort the logging process
    if (!_sdCardMutex.trylock())
        return;
    
    // Open file
    FILE* fp = fopen(_eventLogFileName, "a");
    if (fp == NULL)
    {
        printf ("Event Log ... Filename %s not found\r\n", _eventLogFileName);
    }
    else
    {
        fprintf(fp, timeBuf);
        va_list argptr;
        va_start(argptr, format);
        vfprintf(fp, format, argptr);
        va_end(argptr);
        fprintf(fp, "\r\n");
        fclose(fp);
    }
    
    // Release lock on sd card
    _sdCardMutex.unlock();

}

// Utility function to log data
void Logger::LogData(const char* format, ...)
{
    char fileNameBuf[60];
    strcpy(fileNameBuf, _dataLogFileBase);
    time_t seconds = time(NULL);
    strftime(fileNameBuf+strlen(fileNameBuf), sizeof(fileNameBuf)-strlen(fileNameBuf), "Data_%Y%m%d.txt", localtime(&seconds));
    
    // Obtain lock to access sd card - if unable then abort the logging process
    if (!_sdCardMutex.trylock())
        return;
    
    FILE* fp = fopen(fileNameBuf, "a");
    if (fp == NULL)
    {
        printf ("Data Log ... Filename %s not found\r\n", _eventLogFileName);
    }
    else
    {
        va_list argptr;
        va_start(argptr, format);
        vfprintf(fp, format, argptr);
        va_end(argptr);
        fprintf(fp, "\r\n");
        fclose(fp);
    }
    
    // Release lock on sd card
    _sdCardMutex.unlock();    
}

// Utility function to log data
void Logger::LogDebug(const char* format, ...)
{
    char fileNameBuf[60];
    strcpy(fileNameBuf, _dataLogFileBase);
    time_t seconds = time(NULL);
    strftime(fileNameBuf+strlen(fileNameBuf), sizeof(fileNameBuf)-strlen(fileNameBuf), "Dbg_%Y%m%d.txt", localtime(&seconds));
    
    // Log to file if enabled
    if (_logDebugToFile)
    {
        // Obtain lock to access sd card - if unable then abort the logging process
        if (_sdCardMutex.trylock())
        {
            FILE* fp = fopen(fileNameBuf, "a");
            if (fp == NULL)
            {
                printf ("Data Log ... Filename %s not found\r\n", _eventLogFileName);
            }
            else
            {
                char timeBuf[40];
                time_t seconds = time(NULL);
                strftime(timeBuf, sizeof(timeBuf), "%Y-%m-%d\t%H:%M:%S\t", localtime(&seconds));
                fprintf(fp, timeBuf);        
                va_list argptr;
                va_start(argptr, format);
                vfprintf(fp, format, argptr);
                va_end(argptr);
                fprintf(fp, "\r\n");
            }
            fclose(fp);
    
            // Release lock on sd card
            _sdCardMutex.unlock();    
        }
    }

    // Print to terminal if enabled
    if (_logDebugToConsole)
    {
        va_list argptr;
        va_start(argptr, format);
        vprintf(format, argptr);
        va_end(argptr);
        printf("\r\n");
    }
}
