/**
 * LED Matrix library for http://www.seeedstudio.com/depot/ultrathin-16x32-red-led-matrix-panel-p-1582.html
 * The LED Matrix panel has 32x16 pixels. Several panel can be combined together as a large screen.
 * 
 * Coordinate & Connection (mbed -> panel 0 -> panel 1 -> ...)
 *   (0, 0)                                     (0, 0)
 *     +--------+--------+--------+               +--------+--------+
 *     |   5    |    3   |    1   |               |    1   |    0   |
 *     |        |        |        |               |        |        |<----- mbed
 *     +--------+--------+--------+               +--------+--------+
 *     |   4    |    2   |    0   |                              (64, 16)
 *     |        |        |        |<----- mbed
 *     +--------+--------+--------+
 *                             (96, 32)
 *  Copyright (c) 2013 Seeed Technology Inc.
 *  @auther     Yihui Xiong
 *  @date       Nov 7, 2013
 *  @license    Apache
 */


#ifndef __LED_MATRIX_H__
#define __LED_MATRIX_H__

#include "mbed.h"

const int LED_MATRIX_LEDS_HORIZONTALLY = 32;
const int LED_MATRIX_LEDS_VERTICALLY = 16;
const int LED_MATRIX_MAX_LINES = 2;

class LEDMatrix 
{
public:
    
    LEDMatrix(PinName pinA, PinName pinB, PinName pinC, PinName pinD, PinName pinOE, PinName pinR1, PinName pinSTB, PinName pinCLK);
    
    /**
     * set the display's display buffer and number, the buffer's size must be not less than 512 * number / 8 bytes
     * @param pDisplayBuf    display buffer
     * @param number        panels' number
     */
    void begin(uint8_t *pDisplayBuf, uint16_t width, uint16_t height, uint16_t scrollWidth, 
        uint16_t numLines, int charSeparation = 1, int flashRate = -1, int scrollRate = -1);

    /**
     * draw a point - origin is like a graph with 0,0 at the lower-left corner
     * @param x     x
     * @param y     y
     * @param pixel 0: led off, >0: led on
     */
    void drawPoint(uint16_t x, uint16_t y, uint8_t pixel);

    /**
     * draw a rect - origin is like a graph with 0,0 at the lower-left corner
     * @param (x1, y1)   top-left position
     * @param (x2, y2)   bottom-right position, not included in the rect
     * @param pixel      0: rect off, >0: rect on
     */
    void drawRect(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint8_t pixel);

    /**
     * draw a image
     * @param (x1, y1)   top-left position
     * @param (x2, y2)   bottom-right position, not included in the rect
     * @param pixels     contents, 1 bit to 1 led
     */
    void drawImage(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, uint8_t *image);

    /**
     * turn off 1/16 leds and turn on another 1/16 leds
     */
    void scan();

    void invertColour();

    uint8_t isInvertedColour();

    void on();

    void off();
    
    // Passing lineIdx == -1 to the following functions resets all lines
    // Lines are numbered starting at 0
    void clear(int lineIdx = -1);
    void scroll(int lineIdx, bool scrollLeft);
    void scrollReset(int lineIdx = -1);
    void scrollToPos(int lineIdx, int pos);
    void setupScroll(int lineIdx, int scrollWidth, int scrollInc);

    uint8_t reverseBits(uint8_t b);
    int displayChar(int xPos, int yPos, char ch);
    int displayLargeDigit(int curX, int curY, char ch);
    int displayLine(int lineIdx, const char* line);
    
    void setScrollRate(int rate);
    void setFlashRate(int rate);
    void setAlert(int lineIdx, bool alertOn);

    // Called frequently and regularly to handle effects like scrolling
    void serviceEffects();
    
private:
    bool getRowsToWorkOn(int lineIdx, int &startRow, int &numRows);
    DigitalOut a, b, c, d, oe, r1, stb, clk;
    uint8_t *_pDisplayBuf;
    uint16_t width;
    uint16_t height;
    uint16_t dispBufWidth;
    uint16_t numLines;
    uint8_t  mask;
    bool _isEnabled;
    int _hScrollPos[LED_MATRIX_LEDS_VERTICALLY];
    int _hScrollWidth[LED_MATRIX_LEDS_VERTICALLY];
    bool _isBusy;
    int _charSeparation;
    int _lineScrollInc[LED_MATRIX_MAX_LINES];
    bool _lineAlert[LED_MATRIX_MAX_LINES];
    bool _lineInvert[LED_MATRIX_MAX_LINES];
    int _flashCounter;
    int _flashRate;
    bool _flashState;
    int _scrollCounter;
    int _scrollRate;
};

#endif
