/**
Scoring Device - for generic game scoring
Using Puck BLE MBED library from Nordic
Copyright (C) Nodule.io 2014

 */

#define LOG_LEVEL_DEBUG
#include "Puck.h"
#include "SampleChannel.h"

// This is a singleton for the Nordic Puck library which helps with some BLE init, etc.
Puck* puck = &Puck::getPuck();

// Gatt characteristic and service UUIDs
const UUID SCORING_GATT_SERVICE =               stringToUUID("nod.score1.serv ");
const UUID THRESHOLD_GATT_CHARACTERISTIC =      stringToUUID("nod.score1.thres");
const UUID DIVISOR_GATT_CHARACTERISTIC =        stringToUUID("nod.score1.div  ");
const UUID INTERVAL_US_GATT_CHARACTERISTIC =    stringToUUID("nod.score1.intus");

// Three channels for scoring
const int NUM_SAMPLE_CHANNELS = 3;

// Sample interval (uS)
volatile uint32_t sampleIntervalUs = 10000;

// Interrupt driven ticker to do the sampling
Timeout sampleTimeout;

// Sample Channels
SampleChannel sampleChannels[] =
{
    SampleChannel(P0_1, stringToUUID("nod.score1.samp1"), &logger),
    SampleChannel(P0_2, stringToUUID("nod.score1.samp2"), &logger),
    SampleChannel(P0_3, stringToUUID("nod.score1.samp3"), &logger)    
};

// Timer to avoid repeat sampling
Timer intervalTimer;
int lastTriggerTime = 0;
int lastSampleTime = 0;
const int MIN_MS_BETWEEN_TRIGGERS = 2000;

// Function called in interrupt driven timeout to handle sampling
static volatile int serviceCount = 0;
void SampleService()
{
    // For debug timing
    serviceCount++;
    
    // Service all channel's state machines
    bool isAnyChannelSampling = false;
    for (int chanIdx = 0; chanIdx < NUM_SAMPLE_CHANNELS; chanIdx++)
    {
        sampleChannels[chanIdx].Service();
        if (sampleChannels[chanIdx].IsSampling())
            isAnyChannelSampling = true;
    }
    
    // Check if any channel is being sampled already (if so don't check for triggering)
    if (!isAnyChannelSampling)
    {
        // Check for triggering only if previous trigger was a reasonable time ago
        int curTimerVal = intervalTimer.read_ms();
        // Check for lastTriggerTime < curTimerVal is to handle (not perfectly) overflow/reset of intervalTimer
        if ((lastTriggerTime < curTimerVal) || (curTimerVal - lastTriggerTime > MIN_MS_BETWEEN_TRIGGERS))
        {            
            // Check each channel to see if it's been triggered
            bool anythingTriggered = false;
            for (int chanIdx = 0; chanIdx < NUM_SAMPLE_CHANNELS; chanIdx++)
            {
                if (sampleChannels[chanIdx].CheckTrigger())
                {
                    anythingTriggered = true;
                    LOG_INFO("Triggered\n");
                    break;
                }
            }
            // If any channel has triggered ...
            if(anythingTriggered)
            {
                // Start sampling on all channels
                for (int chanIdx = 0; chanIdx < NUM_SAMPLE_CHANNELS; chanIdx++)
                    sampleChannels[chanIdx].StartSampling();
                // Set timer to disallow repeated readings
                lastTriggerTime = curTimerVal;                
            }
        }
    }
    
    // Request a callback to this function after the sample interval
    sampleTimeout.attach_us(&SampleService, sampleIntervalUs);
}

// ThresholdSet ... BLE characteristic callback
void onThresholdSet(uint8_t* value)
{
    uint16_t threshold = value[0] * 256 + value[1];
    LOG_INFO("Threshold=%d\n", threshold);
    for (int chanIdx = 0; chanIdx < NUM_SAMPLE_CHANNELS; chanIdx++)
        sampleChannels[chanIdx].SetThreshold(threshold);
}

// DivisorSet ... BLE characteristic callback
void onDivisorSet(uint8_t* value)
{
    uint16_t divisor = value[0] * 256 + value[1];
    LOG_INFO("Divisor=%d\n", divisor);
    for (int chanIdx = 0; chanIdx < NUM_SAMPLE_CHANNELS; chanIdx++)
        sampleChannels[chanIdx].SetDivisor(divisor);
}

// Inverval ... BLE characteristic callback
void onIntervalSet(uint8_t* value)
{
    uint32_t intervalUs = (value[0] << 24) + (value[1] << 16) + (value[2] << 8) + value[3];
    LOG_INFO("SampleInterval(uS)=%d\n", intervalUs);
    // Interval timer is restarted in the Ticker callback so just need to store this value
    if (intervalUs <= 1000000)
        sampleIntervalUs = intervalUs;
}

// Main - Setup BLE and service the trigger sampling and BLE
int main(void) 
{
    
    // Set baud rate
    logger.baud(115200);
    
    // Add the Gatt characteristic for samples
    for (int chanIdx = 0; chanIdx < NUM_SAMPLE_CHANNELS; chanIdx++)
    {
        puck->addCharacteristic(
            SCORING_GATT_SERVICE,
            sampleChannels[chanIdx].GetUUID(),
            sampleChannels[chanIdx].GetSamplesLen(),
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);
    }

    // Add the Gatt characteristic for threshold
    puck->addCharacteristic(
            SCORING_GATT_SERVICE,
            THRESHOLD_GATT_CHARACTERISTIC,
            sizeof(uint16_t),
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE);
    puck->onCharacteristicWrite(&THRESHOLD_GATT_CHARACTERISTIC, onThresholdSet);

    // Add the Gatt characteristic for sample divisor
    puck->addCharacteristic(
            SCORING_GATT_SERVICE,
            DIVISOR_GATT_CHARACTERISTIC,
            sizeof(uint16_t),
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE);
    puck->onCharacteristicWrite(&DIVISOR_GATT_CHARACTERISTIC, onDivisorSet);
    
    // Add the Gatt characteristic for sample interval (us)
    puck->addCharacteristic(
            SCORING_GATT_SERVICE,
            INTERVAL_US_GATT_CHARACTERISTIC,
            sizeof(sampleIntervalUs),
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE);
    puck->onCharacteristicWrite(&INTERVAL_US_GATT_CHARACTERISTIC, onIntervalSet);
    
    // Initialize the puck
    puck->init(0xCD01);
    
    // Start timer
    intervalTimer.start();
    
    // Start timeout to service the sampling
    sampleTimeout.attach_us(&SampleService, sampleIntervalUs);

    // Wait for something to be found
    unsigned int lastPuckDriveTime = 0;
    unsigned int driveLoops = 0;
    unsigned int lastDriveLoops = 0;
    unsigned int lastServiceCount = 0;
    while(true)
    {
        // Service the puck
        puck->drive();
        driveLoops++;
        
        // Handle 1 second updates
        unsigned int nowTime = intervalTimer.read_ms();
        if ((nowTime - lastPuckDriveTime >= 1000) || (nowTime < lastPuckDriveTime))
        {
            unsigned int elapsed = nowTime - lastPuckDriveTime;
            LOG_INFO("E%u C%u DC%u DTC%u L%u DL%u\n", elapsed, serviceCount, serviceCount-lastServiceCount, elapsed/(serviceCount-lastServiceCount), driveLoops, driveLoops-lastDriveLoops);
            
            // Check for overflow of timer
            if (nowTime > 100000)
            {
                intervalTimer.reset();
                nowTime = 0;
            }
                
            // Record last timer value
            lastPuckDriveTime = nowTime;
            lastDriveLoops = driveLoops;
            lastServiceCount = serviceCount;
        }
        
        // Check for data ready
        for (int chanIdx = 0; chanIdx < NUM_SAMPLE_CHANNELS; chanIdx++)
        {
            if (sampleChannels[chanIdx].AreSamplesReady())
            {
               // Set the value of the characteristic
                //puck->updateCharacteristicValue(sampleChannels[chanIdx].GetUUID(), sampleChannels[chanIdx].GetSamples(), sampleChannels[chanIdx].GetSamplesLen());
                sampleChannels[chanIdx].StopSampling();
                LOG_INFO("StopSampling\n");
            }
        }
    }
}
