// BLE Service for Watch Time
// To allow a watch device to be told the current time

#ifndef __BLE_WATCHTIME_SERVICE_H__
#define __BLE_WATCHTIME_SERVICE_H__

class WatchTimeService {
public:

    static const int WatchTime_BlockSize = 7;
    const static uint16_t WATCHTIME_SERVICE_UUID              = 0xFE32;
    const static uint16_t WATCHTIME_STATE_CHARACTERISTIC_UUID = 0xFE33;
    const static uint16_t ALARMTIME_STATE_CHARACTERISTIC_UUID = 0xFE34;

    WatchTimeService(BLE &ble, uint8_t* pInitialWatchTime, uint8_t* pInitialAlarmTime) :
        _ble(ble), 
        _watchTimeVal(WATCHTIME_STATE_CHARACTERISTIC_UUID, pInitialWatchTime, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        _alarmTimeVal(ALARMTIME_STATE_CHARACTERISTIC_UUID, pInitialAlarmTime, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)        
    {
        GattCharacteristic *charTable[] = {&_watchTimeVal, &_alarmTimeVal};
        GattService watchTimeService(WatchTimeService::WATCHTIME_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));
        _ble.gattServer().addService(watchTimeService);
    }

    void writeWatchTime(time_t timeToWrite)
    {
        uint8_t watchTime[WatchTimeService::WatchTime_BlockSize];
        unixTimeToWatchTime(timeToWrite, watchTime);
        _ble.gattServer().write(_watchTimeVal.getValueHandle(), watchTime, WatchTime_BlockSize);
    }

    GattAttribute::Handle_t getWatchTimeValueHandle()
    {
        return _watchTimeVal.getValueHandle();
    }
    
    GattAttribute::Handle_t getAlarmTimeValueHandle()
    {
        return _alarmTimeVal.getValueHandle();
    }
    
    static time_t watchTimeToUnixTime(const uint8_t* pWatchTime)
    {
        struct tm tminfo;
        tminfo.tm_year = int(pWatchTime[0])*256 + pWatchTime[1] - 1900;
        tminfo.tm_mon = pWatchTime[2] - 1;
        tminfo.tm_mday = pWatchTime[3];
        tminfo.tm_hour = pWatchTime[4];
        tminfo.tm_min = pWatchTime[5];
        tminfo.tm_sec = pWatchTime[6];
        tminfo.tm_isdst = -1;
        time_t timest = mktime(&tminfo);
        return timest;
    }
    
    static void unixTimeToWatchTime(time_t unixTime, uint8_t* pWatchTime)
    {
        // Convert to localtime
        struct tm * timeinfo;     
        timeinfo = localtime (&unixTime);
        pWatchTime[0] = (timeinfo->tm_year + 1900) / 256;
        pWatchTime[1] = (timeinfo->tm_year + 1900) % 256;
        pWatchTime[2] = timeinfo->tm_mon + 1;
        pWatchTime[3] = timeinfo->tm_mday;
        pWatchTime[4] = timeinfo->tm_hour;
        pWatchTime[5] = timeinfo->tm_min;
        pWatchTime[6] = timeinfo->tm_sec;
    }

private:
    BLE &_ble;
    ReadWriteArrayGattCharacteristic<uint8_t, WatchTime_BlockSize> _watchTimeVal;
    ReadWriteArrayGattCharacteristic<uint8_t, WatchTime_BlockSize> _alarmTimeVal;
};

#endif /* #ifndef __BLE_WATCHTIME_SERVICE_H__ */
