#ifndef __LED_DISPLAY_H__
#define __LED_DISPLAY_H__

#include "nrf51_rtc.h"

// Hour LEDs
const int numHourPins = 5;
DigitalOut hourPins[numHourPins] = { p0, p1, p2, p3, p4 };

// Minute LEDs
const int numMinPins = 6;
DigitalOut minPins[numMinPins] = { p23, p24, p25, p28, p29, p30 };

// Status LEDs
const int numStatusPins = 3;
DigitalOut statusPins[numStatusPins] = { p6, p10, p11 };

// Status bit numbers
const int STATUS_BITS_EXTRA_BIT = 0;
const int STATUS_BITS_SECS_BIT = 1;
const int STATUS_BITS_ALARM_BIT = 2;

void callbackForLEDMuxing();

class LedDisplay
{
public:
    enum DispType {
        DispType_None = 0,
        DispType_CurTime = 1,
        DispType_AlarmTime = 2,
        };
    
    LedDisplay()
    {
        _dispType = DispType_None;
        _timeDisplayStartTime = 0;
        _secsToDisplayFor = 0;
        _timeToShow_Hour = 0;
        _timeToShow_Min = 0;
        _statusBits = 0;
        _curTimeLEDBitPos = 0;
        clear();
    }

    void clear()
    {
        // Clear all LEDs
        for (int i = 0; i < numHourPins; i++)
            hourPins[i] = 0;
        for (int i = 0; i < numMinPins; i++)
            minPins[i] = 0;
        for (int i = 0; i < numStatusPins; i++)
            statusPins[i] = 0;
    }
    
    void start(time_t timeToShow, DispType dispType, int secsToDisplayFor, bool extraInfoBit)
    {
        // Save time to show
        _timeToShow_time_t = timeToShow;
        _dispType = dispType;
        _secsToDisplayFor = secsToDisplayFor;
        if (dispType == DispType_None)
            return;

        // Convert time to binary displayable info (localtime)
        if (((int)_timeToShow_time_t != -1) && ((int)_timeToShow_time_t != 0))
        {
            struct tm * timeinfo;
            timeinfo = localtime (&_timeToShow_time_t);           
            _timeToShow_Hour = timeinfo->tm_hour;
            _timeToShow_Min = timeinfo->tm_min;
        }
        else
        {
            _timeToShow_Hour = 0;
            _timeToShow_Min = 0;
        }
        
        // Status - extra bit
        _statusBits = 0;
        if (extraInfoBit)
            _statusBits |= (1 << STATUS_BITS_EXTRA_BIT);
            
        // Status - alarm bit
        if (dispType == DispType_AlarmTime)
            _statusBits |= (1 << STATUS_BITS_ALARM_BIT);
        else
            _statusBits &= (~(1 << STATUS_BITS_ALARM_BIT));
        
        // Record time we started displaying
        time_t curTime = rtc.time();
        _timeDisplayStartTime = curTime;

        // Ready to display on next callback
        _curTimeLEDBitPos = 0;
        _timerForLEDMuxing.attach(callbackForLEDMuxing, 0.001);

    }
    
    void stop()
    {
        _dispType = DispType_None;
    }
    
    void showNext()
    {
        // Clear LEDs
        clear();

        // Check display type
        if (_dispType == DispType_None)
            return;

        // Get current time
        time_t curTime = rtc.time();
        
        // Stop displaying time after a certain number of seconds
        if (curTime - _timeDisplayStartTime >= _secsToDisplayFor)
        {
            stop();
            return;
        }
    
        // Flash status seconds
        if (_dispType == DispType_CurTime)
        {
            if ((curTime % 2) == 0)
                _statusBits |= (1 << STATUS_BITS_SECS_BIT);
            else
                _statusBits &= (~(1 << STATUS_BITS_SECS_BIT));
        }
        
        // Display binary time
        int mask = 1 << _curTimeLEDBitPos;
        if ((_curTimeLEDBitPos < numHourPins) && ((_timeToShow_Hour & mask) != 0))
            hourPins[_curTimeLEDBitPos] = 1;
        if ((_curTimeLEDBitPos < numMinPins) && ((_timeToShow_Min & mask) != 0))
            minPins[_curTimeLEDBitPos] = 1;
        if ((_curTimeLEDBitPos < numStatusPins) && ((_statusBits & mask) != 0))
            statusPins[_curTimeLEDBitPos] = 1;
        
        // Next bit pos
        _curTimeLEDBitPos++;
        if (_curTimeLEDBitPos > numMinPins)
            _curTimeLEDBitPos = 0;
    
        // Set for another callback
        _timerForLEDMuxing.attach(callbackForLEDMuxing, 0.001);
    }
    
private:

    // Display type
    DispType _dispType;

    // Time to show
    time_t _timeToShow_time_t;
    int _timeToShow_Hour;
    int _timeToShow_Min;
    
    // Time displaying started and time to show for
    time_t _timeDisplayStartTime;
    int _secsToDisplayFor;
    
    // Status bit
    int _statusBits;
    
    // Timeout used to display LEDs
    Timeout _timerForLEDMuxing;
    
    // Cur time disp info
    int _curTimeLEDBitPos;
};

static LedDisplay ledDisplay;

void callbackForLEDMuxing()
{
    ledDisplay.showNext();
}


#endif /* #ifndef __BLE_BUTTON_SERVICE_H__ */