#include "mbed.h"
#include "SpiFlash25.h"
#include "spiffs.h"
#include <string>

// this value represents the number of files you can have open at the same time
// adjust it according to your requirements
#define MAX_CONCURRENT_FDS      4

#define PAGE_SIZE               256
#define SECTOR_SIZE             64*1024
#define MEM_SIZE                2*1024*1024

static u8_t spiffs_work_buf[PAGE_SIZE * 2];
static u8_t spiffs_fds[32 * MAX_CONCURRENT_FDS];
static u8_t spiffs_cache_buf[(PAGE_SIZE + 32) * 4];

static SpiFlash25 flash(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS1);
static spiffs fs;
static spiffs_config cfg;

// glue code between SPI driver and filesystem
int spi_read(unsigned int addr, unsigned int size, unsigned char* data) {
    if (flash.read(addr, size, (char*)data))
        return SPIFFS_OK;
    return -1;
}
int spi_write(unsigned int addr, unsigned int size, unsigned char* data) {
    if (flash.write(addr, size, (const char*)data))
        return SPIFFS_OK;
    return -1;
}
int spi_erase(unsigned int addr, unsigned int size) {
    flash.clear_sector(addr);
    return SPIFFS_OK;
}

int main(void) {
    int ret;
    int handle;
    char data[256];
    char msg[] = "Hello World! The sneaky cat crept around the sleeping dog.\r\n";
    char file[] = "test.txt";
    string sdata;

    // configure the filesystem
    cfg.phys_size = MEM_SIZE;
    cfg.phys_addr = 0;
    cfg.phys_erase_block = SECTOR_SIZE;
    cfg.log_block_size = SECTOR_SIZE;
    cfg.log_page_size = PAGE_SIZE;

    cfg.hal_read_f = &spi_read;
    cfg.hal_write_f = &spi_write;
    cfg.hal_erase_f = &spi_erase;

    Serial pc(USBTX, USBRX);
    pc.baud(115200);

    printf("dragonfly spi flash example started\r\n");

    // erase entire flash
    // THIS WILL ERASE THE ENTIRE FLASH! EVERYTHING ON IT WILL BE LOST AND CANNOT BE RECOVERED!
    //flash.clear_mem();

    // mount the filesystem
    ret = SPIFFS_mount(&fs, &cfg, spiffs_work_buf, spiffs_fds, sizeof(spiffs_fds), spiffs_cache_buf, sizeof(spiffs_cache_buf), NULL);
    if (ret) {
        printf("SPIFFS_mount failed %d - can't continue\r\n", ret);
        return 1;
    }

    // write to the file
    handle = SPIFFS_open(&fs, file, SPIFFS_CREAT | SPIFFS_RDWR | SPIFFS_APPEND, 0);
    if (handle < 0)
        printf("SPIFFS_open failed %d\r\n", SPIFFS_errno(&fs));

    if (handle) {
        ret = SPIFFS_write(&fs, handle, msg, sizeof(msg));
        if (ret < 0)
            printf("SPIFFS_write failed %d\r\n", SPIFFS_errno(&fs));
        else
            printf("wrote %d bytes\r\n", ret);
        SPIFFS_close(&fs, handle);
    }

    // read the current file contents
    spiffs_stat stat;
    memset(&stat, 0, sizeof(stat));
    ret = SPIFFS_stat(&fs, file, &stat);
    if (ret)
        printf("SPIFFS_stat failed %d\r\n", SPIFFS_errno(&fs));
    else
        printf("file size: %d bytes\r\n", stat.size);

    handle = SPIFFS_open(&fs, file, SPIFFS_RDWR, 0);
    if (handle < 0)
        printf("SPIFFS_open failed %d\r\n", SPIFFS_errno(&fs));

    if (handle) {
        while (sdata.size() < stat.size) {
            ret = SPIFFS_read(&fs, handle, data, stat.size - sdata.size() < sizeof(data) ? stat.size - sdata.size() : sizeof(data));
            if (ret < 0) {
                printf("SPIFFS_read failed %d\r\n", SPIFFS_errno(&fs));
                continue;
            }
            printf("read %d bytes\r\n", ret);
            sdata.append(data, ret);
        }

        printf("data [\r\n");
        for (int i = 0; i < sdata.size(); i++)
            printf("%c", sdata[i]);
        printf("\r\n]\r\n");

        SPIFFS_close(&fs, handle);
    }

    printf("dragonfly spi flash example finished\r\n\r\n");

    return 0;
}