/* This is an example program which reads in the 3-axis acceleration, pressure, and temperature data from
 * a FRDM-FXS-MULTI sensor board. It then uses an MTSAS SocketModem Shield board to send the read data over
 * a cellular connection to Aeris AerCloud using an MQTT client protocol.
 */

#include "mbed.h"
#include "mtsas.h"
#include "PubSubClient.h"
#include "x_cube_mems.h"

//
// PLEASE READ THIS!
//
// Example was created for the following hardware:
//   ST Nucleo F401RE   http://developer.mbed.org/platforms/ST-Nucleo-F401RE/
//   ST Sensor Board  http://developer.mbed.org/teams/ST-Americas-mbed-Team/wiki/Getting-Started-with-Nucleo-Sensors
//   MultiTech Socket Modem Shield: http://developer.mbed.org/components/Multi-Tech-SocketModem-Arduino-Shield-MT/
//   MultiTech MTSMC-H5 GSM Socket Modem http://www.multitech.com/models/92503252LF#gsc.tab=0
//
// To get the sample running, you'll need to fill in the following parameters below
//   Your cellular provider's APN: _APN
//   AerCloud API Key: _AERCLOUD_API_APIKEY
//   AerCloud Account ID: _AERCLOUD_ACCOUNT_ID
//   AerCloud Container: _ACERCLOUD_CONTAINER
//
// The AerCloud container needa a Data Model with the following schema:
//    accel_x  FLOAT
//    accel_y  FLOAT 
//    accel-Z  FLOAT
//    pressure FLOAT
//    temperature FLOAT
//
// To check if data has made it in the container, create a long polling subscription in AerPort and they use the following url in your browswer
//
// http://longpoll.aercloud.aeris.com/v1/1/containers/subscriptions/<subscrption_name>/notificationChannels/longPoll?apiKey=<api_key>
//
// You should see the something that looks like this in the browser:
// {"sclContentInstances":[{"sclId":"nucleo-0001","containerId":"Nucleo_Test","contentInstance":{"id":"a40c8e60-8248-11e4-8b38-0677f0dfdf5e","contentSize":90,"creationTime":1418420922950,"content":{"contentType":"application/json","contentTypeBinary":"{\"x\":0.005615,\"y\":-0.041260,\"z\":1.015137,\"pressure\":101098.500000,\"temperature\":25.125000}"}}},
//



char _APN[] = "Set_carrier_APN";   

char _AERCLOUD_API_KEY[] ="_Click_On_KEY_AerCloud_Tab";
char _AERCLOUD_ACCOUNT_ID[] = "_Company_Number_Top_of_Page";     
char _AERCLOUD_CONTAINER[] = "_From_Container_Page";                 
char _AERCLOUD_DEVICE_ID[] = "_Random_Title";                     


char _host[] = "mqtt.aercloud.aeris.com";
int _port = 1883;

#define DATA_INTERVAL 30

void callback(char* topic, char* payload, unsigned int len) {
    logInfo("topic: [%s]\r\npayload: [%s]", topic, payload);
}

DigitalOut myled(LED1);
Serial pc(SERIAL_TX, SERIAL_RX);
volatile float TEMPERATURE_Value_C;
volatile float TEMPERATURE_Value_F;
volatile float HUMIDITY_Value;
volatile float PRESSURE_Value;
volatile AxesRaw_TypeDef MAG_Value;
volatile AxesRaw_TypeDef ACC_Value;
volatile AxesRaw_TypeDef GYR_Value;

int main() {
    printf("Hello FUTURE!\r\n");
    MTSLog::setLogLevel(MTSLog::TRACE_LEVEL);
    
    // for Nucleo boards
        MTSSerialFlowControl io(D8, D2, D3, D13);    // Modified Flow control for ST Sensor Board  D13
    io.baud(115200);
    
    Cellular* radio = CellularFactory::create(&io);
    if (! radio) {
        logFatal("failed to create Cellular object - exiting");
        return 1;
    }
    
    radio->configureSignals(D11,D7,RESET);          // Modified Flow Control for ST Sensor Board  D11
    Transport::setTransport(radio);                 // Required to control Cell Radio vs WiFi solution
    
    while (radio->setApn(_APN) != MTS_SUCCESS) {
        logError("failed to set APN [%s]", _APN);
        wait(2);
    }
    
    while (! radio->connect()) {
        logError("failed to bring up PPP link");
        wait(2);
    }
    
    printf("Signal Strength: %d\n\r", radio->getSignalStrength()); //Check the signal strength should be above 8
    
    // If you suspect a connectivity issue, uncomment the code below and if ping works.  If you are not getting a
    //  valid ping, there's a connectivity problem.  First step is to verify you've got the right APN set
    //
    // Try pinging default server "8.8.8.8" (Google's DNS)
    //  int ping_valid = 0;
    //  while (ping_valid == 0) {
    //      ping_valid = radio->ping();
    //      printf("Ping Valid: %s\n\r", ping_valid ? "true" : "false");
    //    
    //      if (ping_valid == 0) {
    //          wait(3);
    //          printf("wait 33");
    //          wait(33);
            
    //    }
    //}
    
    PubSubClient mqtt(_host, _port, callback);
    
    char buf[128];
    
    static X_CUBE_MEMS *mems_expansion_board = X_CUBE_MEMS::Instance();
  
        
        /* Get  data from all sensors */
        mems_expansion_board->hts221.GetTemperature((float *)&TEMPERATURE_Value_C);
        mems_expansion_board->hts221.GetHumidity((float *)&HUMIDITY_Value);
        mems_expansion_board->lps25h.GetPressure((float *)&PRESSURE_Value);
        mems_expansion_board->lis3mdl.GetAxes((AxesRaw_TypeDef *)&MAG_Value);
        mems_expansion_board->lsm6ds0.Acc_GetAxes((AxesRaw_TypeDef *)&ACC_Value);
        mems_expansion_board->lsm6ds0.Gyro_GetAxes((AxesRaw_TypeDef *)&GYR_Value);
        
        TEMPERATURE_Value_F = (TEMPERATURE_Value_C * 1.8f) + 32.0f;
        pc.printf("Temperature:\t\t %f C\r\n", TEMPERATURE_Value_C );
//        pc.printf("Humidity:\t\t %f%%\r\n", HUMIDITY_Value);
        pc.printf("Pressure:\t\t %f hPa\r\n", PRESSURE_Value); 
//        pc.printf("Magnetometer (mGauss):\t mX: %d, Y: %d, mZ: %d\r\n", MAG_Value.AXIS_X, MAG_Value.AXIS_Y, MAG_Value.AXIS_Z);
        pc.printf("Accelerometer (mg):\t X: %d, Y: %d, Z: %d\r\n", ACC_Value.AXIS_X, ACC_Value.AXIS_Y, ACC_Value.AXIS_Z);
//        pc.printf("Gyroscope (mdps):\t X: %d, Y: %d, Z: %d\r\n", GYR_Value.AXIS_X, GYR_Value.AXIS_Y, GYR_Value.AXIS_Z);
        pc.printf("\r\n");
        

    while (true) {
       if (! mqtt.connect(_AERCLOUD_DEVICE_ID, _AERCLOUD_ACCOUNT_ID, _AERCLOUD_API_KEY)) {
            logError("failed to connect to AerCloud Server");
            wait(5);
            continue;
        }

//  get new data
        myled = 1; // LED is ON
        wait(0.2); // 200 ms
        myled = 0; // LED is OFF
        
        mems_expansion_board->hts221.GetTemperature((float *)&TEMPERATURE_Value_C);
//        mems_expansion_board->hts221.GetHumidity((float *)&HUMIDITY_Value);
        mems_expansion_board->lps25h.GetPressure((float *)&PRESSURE_Value);
//        mems_expansion_board->lis3mdl.GetAxes((AxesRaw_TypeDef *)&MAG_Value);
        mems_expansion_board->lsm6ds0.Acc_GetAxes((AxesRaw_TypeDef *)&ACC_Value);
//        mems_expansion_board->lsm6ds0.Gyro_GetAxes((AxesRaw_TypeDef *)&GYR_Value);
        pc.printf("Accelerometer (mg):\t X: %d, Y: %d, Z: %d\r\n\n", ACC_Value.AXIS_X, ACC_Value.AXIS_Y, ACC_Value.AXIS_Z); 
              
        snprintf(buf, sizeof(buf), "{\"xxx\":%d,\"yyy\":%d,\"zzz\":%d,\"pressure\":%f,\"temperature\":%f}",ACC_Value.AXIS_X, ACC_Value.AXIS_Y, ACC_Value.AXIS_Z, PRESSURE_Value, TEMPERATURE_Value_C);
        logInfo("publishing: [%s]", buf);
        if (! mqtt.publish(_AERCLOUD_CONTAINER, buf)) {
            logError("failed to publish: [%s]", buf);
        }
        wait(1);
        mqtt.loop();
        mqtt.disconnect();
        wait(DATA_INTERVAL);
        
    }
    
}

