/** Combination of MultiTech HTTPS Example using json with NXP Freescale sensor board and IBM Bluemix
 *
 * Configures the Sensor board for Accelerometer, cellular radio, brings up the cellular link,
 * and does HTTPS POST requests.
 * To do HTTPS requests with a certain server, the root certificate used to validate that server's certificate must be installed. See ssl_certificates.h for information on how to get the proper root certificate.
 *
 *
  *
 * The following hardware is required to successfully run this program:
 *   - MultiTech UDK2 (4" square white PCB with Arduino headers, antenna
 *     connector, micro USB ports, and 40-pin connector for Dragonfly)
 *   - MultiTech Dragonfly (1"x2" green PCB with Telit radio)
 *   - Freescale (NXP) Sensor Board
 *
 * What this program does:
 *   - reads data from sensor
 *   - prints all sensor data to debug port on a periodic basis
 *   - sends data to BlueMix
 *   - All data is sent to a specific location determined by the student login.
 *   - BlueMix cloud platform (user must create own account and configure a device
 *
 * Setup:
 *   - Correctly insert SIM card into Dragonfly
 *   - Seat the Dragonfly on the UDK2 board
 *   - Connect an antenna to the connector on the Dragonfly labled "M"
 *   - Stack the Base Shield on the UDK2 Arduino headers
 *   - Stack the MEMs board on top of the Base Shield
 *   - Plug in the power cable
 *   - Plug a micro USB cable into the port below and slightly to the
 *     left of the Dragonfly near the RF Connector (NOT the port on the Dragonfly)
 *
 * Go have fun and make something cool!
 *
 ************************************************************************/

#include "mbed.h"
#include "mtsas.h"
#include "ssl_certificates.h"
#include <string>   // added for string manipulation
#include <sstream>
#include "FXAS21002.h"
#include "FXOS8700.h"

// Initialize pins for I2C communication for sensors. Set jumpers J6,J7 in FRDM-STBC-AGM01 board accordingly.
FXOS8700 accel(D14,D15);
FXOS8700 mag(D14,D15);
FXAS21002 gyro(D14,D15);


char streamAcc[] = "acc_rms"; // Stream you want to push to
char streamMag[] = "mag_rms"; // Stream you want to push to
char streamGyr[] = "gyr_rms"; // Stream you want to push to


//*****************************************************************************************************************************************************
DigitalOut Led1Out(LED1);


// This line controls the regulator's battery charger.
// BC_NCE = 0 enables the battery charger
// BC_NCE = 1 disables the battery charger
DigitalOut bc_nce(PB_2);

bool init_mtsas();
char* httpResToStr(HTTPResult res);

// The MTSSerialFlowControl object represents the physical serial link between the processor and the cellular radio.
mts::MTSSerialFlowControl* io;
// The Cellular object represents the cellular radio.
mts::Cellular* radio;

// An APN is required for GSM radios.
static const char apn[] = "";

bool radio_ok = false;

bool init_mtsas();
char* httpResToStr(HTTPResult res);

/****************************************************************************************************
// main
 ****************************************************************************************************/

int main()
{

    accel.accel_config();
    mag.mag_config();
    gyro.gyro_config();

    float accel_data[3];
    //float mag_data[3];
    //float gyro_data[3];

    printf("Begin Data Acquisition from FXOS8700 and FXAS21002....\r\n\r\n");
    wait(0.5);

    // Disable the battery charger unless a battery is attached.
    bc_nce = 1;

    // Change the baud rate of the debug port from the default 9600 to 115200.
    Serial debug(USBTX, USBRX);
    debug.baud(115200);

    //Sets the log level to INFO, higher log levels produce more log output.
    //Possible levels: NONE, FATAL, ERROR, WARNING, INFO, DEBUG, TRACE
    mts::MTSLog::setLogLevel(mts::MTSLog::TRACE_LEVEL);


//****************************************************************************************************/
// Initialization Radio Section **********************************************************
//****************************************************************************************************/
    logInfo("initializing cellular radio");
    radio_ok = init_mtsas();
    if (! radio_ok) {
        while (true) {
            logError("failed to initialize cellular radio");
            wait(1);
        }
    }

    logInfo("setting APN");
    if (radio->setApn(apn) != MTS_SUCCESS)
        logError("failed to set APN to \"%s\"", apn);
    logInfo("APN set successful");

    Timer post_timer;
    post_timer.start();
    static int post_interval_ms = 500; //************* I don't want to wait 30 seconds ************************/
    int timeStamp;
    int countingLoop = 0;

    logInfo("Entering loop");
    while (countingLoop < 5 ) {
        if (post_timer.read_ms() > post_interval_ms ) {            // can this be changed to seconds?
            timeStamp = post_timer.read_ms();
            logDebug("timer read %d", timeStamp);
            logDebug("timer value %d",  post_interval_ms );
            logDebug("loop count value %d",  countingLoop );

            accel.acquire_accel_data_g(accel_data);
            float dataX = accel_data[0];
            float dataY = accel_data[1];
            float dataZ = accel_data[2];

            logDebug("\r\nPosting Accel Readings X: %f Y: %f  Z: %f \r\n",dataX,dataY,dataZ);

            logDebug("https://quickstart.internetofthings.ibmcloud.com");

            //http_tx.clear();

            logInfo("bringing up the link");
            if (! radio->connect()) {
                logError("failed to bring up the link");
                //return 0;
            } else {

                // HTTPClient object used for HTTP requests.
                HTTPClient http;

                // Enable strict certificate validation.
                http.setPeerVerification(VERIFY_PEER);

                // Load certificates defined in ssl_certificates.h.
                // See comments in ssl_certificates.h for information on how to get and format root certificates.
                if (http.addRootCACertificate(ssl_certificates) != HTTP_OK)
                    logError("loading SSL certificates failed");

                char http_rx_buf[1024];
                char http_tx_buf[1024];

                memset(http_tx_buf, 0, sizeof(http_tx_buf));
                memset(http_rx_buf, 0, sizeof(http_rx_buf));
                snprintf(http_tx_buf, sizeof(http_tx_buf), "{ \"dataX\": \"%f\" , \"dataY\": \"%f\" ,\"dataZ\": \"%f\" }", dataX,dataY,dataZ);
                logDebug("%s",http_tx_buf);
                HTTPResult res;

                // IHTTPDataIn object - will contain data received from server.
                HTTPText http_rx(http_rx_buf, sizeof(http_rx_buf));

                // IHTTPDataOut object - contains data to be posted to server.
                // HTTPJson automatically adds the JSON content-type header to the request.
                HTTPJson http_tx(http_tx_buf, strlen(http_tx_buf)+1);

                // Make a HTTP POST request to http://httpbin.org/
//                res = http.post("http://quickstart.internetofthings.ibmcloud.com/api/v0002/device/types/dragonflytype/devices/dragonfly22/events/myEvent", http_tx, &http_rx);
                res = http.post("http://quickstart.internetofthings.ibmcloud.com/api/v0002/device/types/dragonflytype/devices/ REPLACE WITH DEVICEID /events/myEvent", http_tx, &http_rx);
                if (res != HTTP_OK)
                    logError("HTTPS POST to Bluemix failed [%d][%s]", res, httpResToStr(res));
                else
                    logInfo("HTTPS POST to Bluemix succeeded [%d]\r\n%s", http.getHTTPResponseCode(), http_rx_buf);

                //logInfo("finished - bringing down link");
                //radio->disconnect();
                post_timer.reset();
                countingLoop +=1;
            }
        }

        //return 0;
    }
    radio->disconnect();
    timeStamp = post_timer.read_ms();
    logInfo("loop timer = %d", timeStamp);
    logInfo("\r\n\n\nEnd Of Line\r\n");
}

bool init_mtsas()
{
    io = new mts::MTSSerialFlowControl(RADIO_TX, RADIO_RX, RADIO_RTS, RADIO_CTS);
    if (! io)
        return false;

    // radio default baud rate is 115200
    io->baud(115200);
    radio = mts::CellularFactory::create(io);
    if (! radio)
        return false;

    // Transport must be set properly before any TCPSocketConnection or UDPSocket objects are created
    Transport::setTransport(radio);

    return true;
}

char* httpResToStr(HTTPResult res)
{
    switch(res) {
        case HTTP_PROCESSING:
            return "HTTP_PROCESSING";
        case HTTP_PARSE:
            return "HTTP_PARSE";
        case HTTP_DNS:
            return "HTTP_DNS";
        case HTTP_PRTCL:
            return "HTTP_PRTCL";
        case HTTP_NOTFOUND:
            return "HTTP_NOTFOUND";
        case HTTP_REFUSED:
            return "HTTP_REFUSED";
        case HTTP_ERROR:
            return "HTTP_ERROR";
        case HTTP_TIMEOUT:
            return "HTTP_TIMEOUT";
        case HTTP_CONN:
            return "HTTP_CONN";
        case HTTP_CLOSED:
            return "HTTP_CLOSED";
        case HTTP_REDIRECT:
            return "HTTP_REDIRECT";
        case HTTP_OK:
            return "HTTP_OK";
        default:
            return "HTTP Result unknown";
    }
}

