#include <mbed.h>
#include <Sockets.h>
#include <mbedNetIF.h>
#include <ARP.h>
#include <ICMPv4.h>
#include <UDPv4.h>
#include <Debug.h>


#define NET_TIMER_PERIOD        1
#define BUFFER_LEN              64


Serial              pc(USBTX, USBRX); // tx, rx
Ticker              netTicker, echoTicker;
Ethernet_Addr_t     myMAC =			{0x00, 0x10, 0x33, 0x2e, 0x18, 0x7f};
IPv4_Addr_t         ip = 			{192, 168, 0, 2},
                    netmask = 		{255, 255, 255, 0},
                    gateway = 		{192, 168, 0, 1},
                    broadcastNet =	{192, 168, 0, 255},
                    broadcastAll =	{255, 255, 255, 255};
NetIF_t             *mbedNetIF;
Socket_t            udpSocket;
Socket_AddrIn_t     localAddress, senderAddress, broadcastAddress;


void NetTicker(void)
{
    NetIF_ProcessTimers(NET_TIMER_PERIOD);
}


void EchoTicker(void)
{
    char            buffer[BUFFER_LEN];
    static int32_t  counter = 0;
    
    sprintf(buffer, "counter = %d\n", counter++);
    Sockets_SendTo(udpSocket, (uint8_t *)buffer, strlen(buffer), 0, (Socket_Addr_t *)&broadcastAddress, broadcastAddress.len);
}


void DisplayHelp(void)
{
	printf("\r\nCommands:\r\n");
	printf("  d:   display ARP cache\r\n");
	printf("  f:   flush ARP cache\r\n");
	printf("  g:   send ARP request for gateway\r\n");
	printf("  h:   display these short help lines\r\n");
	printf("\r\n");
}


int main() 
{
    int32_t     byteCount, senderAddressLength;
    char        buffer[BUFFER_LEN], c;

    // serial output speed to 115200 baud
    pc.baud(115200);
	setbuf(stdout, NULL);

    // Clears screen and display a welcome message
    printf("%cStarting!\r\n", 12);
    
    // Set debug levels to have the NetIF networking layer display the interface status
    Debug_SetMasks(DEBUG_MODULE_NETIF, DEBUG_LEVEL_ALL  & ~(DEBUG_LEVEL_VERBOSE0 | DEBUG_LEVEL_VERBOSE1 | DEBUG_LEVEL_VERBOSE2));
    
    mbedNetIF = NetIF_RegisterInterface(&ip, &netmask, &gateway, &mbedNetIF_Driver, (void *)&myMAC);    // Registers the ethernet drivers as a network interface
    
    ethernet.RegisterProtocol(&arp);                    // We need ARP over ethernet to resolve IP addresses with MAC addresses
    ethernet.RegisterProtocol(&ipv4);                   // We need IPv4 (over ethernet) of course
    ipv4.RegisterProtocol(&icmpv4);                     // Then ICMPv4 over IPv4 to answer ping requests
    ipv4.RegisterProtocol(&udpv4);                      // Next let's register UDPv4 over IPv4
    udpv4.RegisterAPI(&sockets);                        // Finally let's register the Sockets API (it is available for UDP at the moment)

    ARP_AddStaticEntry(mbedNetIF, broadcastNet, &ethernet_Addr_Broadcast); // We add a static entry to the local network broadcast address to the ARP cache
    ARP_AddStaticEntry(mbedNetIF, broadcastAll, &ethernet_Addr_Broadcast); // We add a static entry to the GLOBAL broadcast address to the ARP cache

    netTicker.attach(&NetTicker, NET_TIMER_PERIOD);     // A 1 second periodic timer to process network house keeping routines (like expiration in the ARP cache, timeouts for TCP in the future, ...)
    
    NetIF_Up(mbedNetIF);                                // Finally we bring the network interface up
    
    udpSocket = Sockets_Open(AF_INET, SOCK_DGRAM, 0);   // Create an UDP socket
    
    localAddress.family = AF_INET;
    localAddress.len = sizeof(Socket_AddrIn_t);
    localAddress.port = htons(12345);
    localAddress.address = ipv4_Addr_Any;
    
    broadcastAddress.family = AF_INET;
    broadcastAddress.len = sizeof(Socket_AddrIn_t);
    broadcastAddress.port = htons(54321);
    broadcastAddress.address = broadcastNet;

    Sockets_Bind(udpSocket, (Socket_Addr_t *)&localAddress, localAddress.len);  // We bind the socket to port 12345 and any local IP address

    echoTicker.attach(&EchoTicker, 2);                  // Every two seconds, broadcast an UDP message to port 54321

    // Then do whatever you need for your application. here we simply check if we have received a datagram from the udpSocket and display its informations
    senderAddressLength = sizeof(Socket_AddrIn_t);
	
	DisplayHelp();
		
	printf(">");
	
    while(true)
    {
        byteCount = Sockets_RecvFrom(udpSocket, (uint8_t *)buffer, BUFFER_LEN, 0, (Socket_Addr_t *)&senderAddress, &senderAddressLength);
        if (byteCount > 0)
        {
            buffer[byteCount] = '\0';
            printf("We have received '%s' (%d bytes) from %d.%d.%d.%d:%d\r\n",
                buffer,
                byteCount,
                senderAddress.address.IP0,
                senderAddress.address.IP1,
                senderAddress.address.IP2,
                senderAddress.address.IP3,
                ntohs(senderAddress.port)
                );
        }
		
		if (pc.readable())
		{
			c = pc.getc();
			printf("%c", c);
			switch(c)
			{
				case 'd':
					ARP_DisplayCache();
					printf("\r\n>");
					break;

				case 'f':
					ARP_FlushCache(False);
					printf("\r\n>");
					break;
					
				case 'g':
					ARP_ResolveIPv4Address(mbedNetIF, gateway, NULL);
					printf("\r\n>");
					break;
					
				case 'h':
					DisplayHelp();
					printf("\r\n>");
					break;
			}
		}
	}
}
