/*
 * $Id: Ethernet.c 29 2011-06-11 14:53:08Z benoit $
 * $Author: benoit $
 * $Date: 2011-06-11 16:53:08 +0200 (sam., 11 juin 2011) $
 * $Rev: 29 $
 * 
 * 
 * 
 * 
 * 
 */
 
#include "Ethernet.h"
#include "Debug.h"
#include <string.h>


#define    DEBUG_CURRENT_MODULE_NAME    "Ethernet"
#define    DEBUG_CURRENT_MODULE_ID        DEBUG_MODULE_ETHERNET


static void Init(void);
static int32_t RegisterProtocol(Protocol_Handler_t *protocolHandler);
static int32_t RegisterAPI(Net_API_t *api);
static void Handler(NetIF_t *netIF, NetPacket_t *packet);


static Protocol_Handler_t    *protocolHandlerTable[ETHERNET_PROTOCOL_MAX_COUNT];
static int32_t                protocolHandlerCount = 0;

const Ethernet_Addr_t        ethernet_Addr_Broadcast =     {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
const Ethernet_Addr_t        ethernet_Addr_Null =         {0, 0, 0, 0, 0, 0};


Protocol_Handler_t ethernet = 
{ 
    PROTOCOL_INDEX_NOT_INITIALIZED,     /* Always PROTOCOL_INDEX_NOT_INITIALIZED at initialization */
    Protocol_ID_Ethernet,               /* Protocol ID */
    PROTOCOL_NUMBER_NONE,               /* Protocol number */
    Init,                               /* Protocol initialisation function */
    Handler,                     		/* Protocol handler */
    RegisterProtocol,                   /* Protocol registration function */
    RegisterAPI,                        /* API registration function */
};


static Net_API_t    *netAPITable[NET_API_PER_PROTOCOL_MAX_COUNT];
static int32_t        netAPICount = 0;


static void Init(void)
{
    DEBUG_MODULE(DEBUG_LEVEL_INFO, ("Initializing ethernet layer"));
    memset(protocolHandlerTable, 0, sizeof(protocolHandlerTable));
    protocolHandlerCount = 0;
    memset(netAPITable, 0, sizeof(netAPITable));
    netAPICount = 0;
}


static int32_t RegisterProtocol(Protocol_Handler_t *protocolHandler)
{
    int32_t                            result = 0;
    
    if (protocolHandlerCount >= ETHERNET_PROTOCOL_MAX_COUNT)
    {
        DEBUG_SOURCE(DEBUG_LEVEL_ERROR, ("Too many protocols"));
        result = -1;
        mbedNet_LastError = mbedNetResult_TooManyRegisteredProtocols;
        goto Exit;
    }
    
    protocolHandlerTable[protocolHandlerCount] = protocolHandler;
    protocolHandler->index = protocolHandlerCount;
    protocolHandler->Init();
    
    DEBUG_MODULE(DEBUG_LEVEL_INFO, ("Registered protocol %04X ethernet/%s",
        ntohs(protocolHandler->protocolNumber),
        protocol_IDNames[protocolHandler->protocolID]
    ));
    
    protocolHandlerCount++;
        
Exit:
    return result;
}


static int32_t RegisterAPI(Net_API_t *netAPI)
{
    DEBUG_MODULE(DEBUG_LEVEL_INFO, ("Registering %s API", api_IDNames[netAPI->apiID]));
    netAPI->InitAPI();
    netAPITable[netAPICount] = netAPI;
    netAPICount++;
    return -1;
}


void Handler(NetIF_t *netIF, NetPacket_t *packet)
{
    int32_t				protocolIndex, index;
    Ethernet_Proto_t	protocolNumber;
    Protocol_Handler_t	*protocolHandler;
    Ethernet_Header_t	*ethernetHeader;
    
	ethernetHeader = (Ethernet_Header_t *)packet->data;
    protocolNumber = ethernetHeader->protocol;
    DEBUG_MODULE(DEBUG_LEVEL_INFO, ("frame of %d bytes for protocol %04X (payload + %02d)", packet->length, ntohs(protocolNumber), sizeof(Ethernet_Header_t)));
    //Debug_DumpBufferHex(packet, length);
    
    /* Process API if any */
    for (index = 0; index < netAPICount; index++)
    {
        netAPITable[index]->Hook(netIF, Protocol_ID_Ethernet, packet);
    }
    
	for (protocolIndex = 0; protocolIndex < protocolHandlerCount; protocolIndex++)
    {
        protocolHandler = protocolHandlerTable[protocolIndex];
        if (protocolHandler->protocolNumber == protocolNumber)
        {
			DEBUG_SOURCE(DEBUG_LEVEL_VERBOSE0, ("'%s' frame of %d bytes", protocol_IDNames[protocolHandler->protocolID], packet->length));
            NetIF_ProtoPush(packet, sizeof(Ethernet_Header_t), Protocol_ID_Ethernet);
            protocolHandler->HandlePacket(netIF, packet);
            break;
        }
    }
    return;
}


