/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: Target board general functions implementation

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Miguel Luis and Gregory Cristian
*/
#ifndef __BOARD_H__
#define __BOARD_H__

#include "Config.h"
#include "mbed.h"
#include "system/timer.h"
#include "debug.h"
#include "system/utilities.h"
#include "sx1272-hal.h"
#include "gps.h"
#include "mpl3115a2.h"
#include "mma8451q.h"

#define LOW_BAT_THRESHOLD   3450 // mV

extern DigitalOut RedLed;
extern DigitalOut GreenLed;
extern DigitalOut YellowLed;
extern DigitalOut UsrLed;
extern GPS Gps;
extern MPL3115A2 Mpl3115a2;
extern MMA8451Q Mma8451q;

extern DigitalIn PC0; // Used for Push button application demo

extern SX1272MB2xAS Radio;

/*!
 * Board versions
 */
typedef enum
{
    BOARD_VERSION_NONE = 0,
    BOARD_VERSION_2,
    BOARD_VERSION_3,
}BoardVersion_t;

/*!
 * \brief Disable interrupts
 *
 * \remark IRQ nesting is managed
 */
void BoardDisableIrq( void );

/*!
 * \brief Enable interrupts
 *
 * \remark IRQ nesting is managed
 */
void BoardEnableIrq( void );

/*!
 * \brief Initializes the target board peripherals.
 */
void BoardInit( void );

/*!
 * \brief Measure the Battery voltage
 *
 * \retval value  battery voltage in millivolts
 */
uint32_t BoardGetBatteryVoltage( void );

/*!
 * \brief Get the current battery level
 *
 * \retval value  battery level [  0: USB,
 *                                 1: Min level,
 *                                 x: level
 *                               254: fully charged,
 *                               255: Error]
 */
uint8_t BoardGetBatteryLevel( void );

/*!
 * Returns a pseudo random seed generated using the MCU Unique ID
 *
 * \retval seed Generated pseudo random seed
 */
uint32_t BoardGetRandomSeed( void );

/*!
 * \brief Gets the board 64 bits unique ID
 *
 * \param [IN] id Pointer to an array that will contain the Unique ID
 */
void BoardGetUniqueId( uint8_t *id );

/*!
 * \brief Get the board version
 *
 * \retval value  Version [0: MOTE_VERSION_NONE, 
 *                         1: MOTE_VERSION_2,
 *                         2: MOTE_VERSION_3]
 */
BoardVersion_t BoardGetVersion( void );

/*!
 * \brief Generates Lower 32 bits of DEVEUI using 96 bits unique device ID 
 *
 * \param [IN] id Pointer to an array that will contain the Unique ID
 */
void BoardGetDevEUI( uint8_t *id );

#endif // __BOARD_H__
