// Simple USBHost for FRDM-KL46Z
#pragma once
#include "mbed.h"
#include "USBHALHost.h"
#include "USBDeviceConnected.h"
#include "IUSBEnumerator.h"
#include "USBHostConf.h"
#include "USBEndpoint.h"

class USBHost : public USBHALHost {
public:
    /**
    * Static method to create or retrieve the single USBHost instance
    */
    static USBHost* getHostInst();

    /**
    * Control read: setup stage, data stage and status stage
    *
    * @param dev the control read will be done for this device
    * @param requestType request type
    * @param request request
    * @param value value
    * @param index index
    * @param buf pointer on a buffer where will be store the data received
    * @param len length of the transfer
    *
    * @returns status of the control read
    */
    USB_TYPE controlRead(USBDeviceConnected * dev, uint8_t requestType, uint8_t request, uint32_t value, uint32_t index, uint8_t * buf, uint32_t len);

    /**
    * Control write: setup stage, data stage and status stage
    *
    * @param dev the control write will be done for this device
    * @param requestType request type
    * @param request request
    * @param value value
    * @param index index
    * @param buf pointer on a buffer which will be written
    * @param len length of the transfer
    *
    * @returns status of the control write
    */
    USB_TYPE controlWrite(USBDeviceConnected * dev, uint8_t requestType, uint8_t request, uint32_t value, uint32_t index, uint8_t * buf, uint32_t len);
    /**
    * Bulk read
    *
    * @param dev the bulk transfer will be done for this device
    * @param ep USBEndpoint which will be used to read a packet
    * @param buf pointer on a buffer where will be store the data received
    * @param len length of the transfer
    * @param blocking if true, the read is blocking (wait for completion)
    *
    * @returns status of the bulk read
    */
    USB_TYPE bulkRead(USBDeviceConnected * dev, USBEndpoint * ep, uint8_t * buf, uint32_t len, bool blocking = true);

    /**
    * Bulk write
    *
    * @param dev the bulk transfer will be done for this device
    * @param ep USBEndpoint which will be used to write a packet
    * @param buf pointer on a buffer which will be written
    * @param len length of the transfer
    * @param blocking if true, the write is blocking (wait for completion)
    *
    * @returns status of the bulk write
    */
    USB_TYPE bulkWrite(USBDeviceConnected * dev, USBEndpoint * ep, uint8_t * buf, uint32_t len, bool blocking = true);

    /**
    * Interrupt read
    *
    * @param dev the interrupt transfer will be done for this device
    * @param ep USBEndpoint which will be used to write a packet
    * @param buf pointer on a buffer which will be written
    * @param len length of the transfer
    * @param blocking if true, the read is blocking (wait for completion)
    *
    * @returns status of the interrupt read
    */
    USB_TYPE interruptRead(USBDeviceConnected * dev, USBEndpoint * ep, uint8_t * buf, uint32_t len, bool blocking = true);

    /**
    * Interrupt write
    *
    * @param dev the interrupt transfer will be done for this device
    * @param ep USBEndpoint which will be used to write a packet
    * @param buf pointer on a buffer which will be written
    * @param len length of the transfer
    * @param blocking if true, the write is blocking (wait for completion)
    *
    * @returns status of the interrupt write
    */
    USB_TYPE interruptWrite(USBDeviceConnected * dev, USBEndpoint * ep, uint8_t * buf, uint32_t len, bool blocking = true);

    /**
    * Enumerate a device.
    *
    * @param dev device which will be enumerated
    *
    * @returns status of the enumeration
    */
    USB_TYPE enumerate(USBDeviceConnected * dev, IUSBEnumerator* pEnumerator);

    /**
    * Get a device
    *
    * @param index index of the device which will be returned
    *
    * @returns pointer on the "index" device
    */
    USBDeviceConnected * getDevice(uint8_t index) {
        return (index < DeviceLists_count) ? DeviceLists[index] : NULL;
    }

    /**
     *  register a driver into the host associated with a callback function called when the device is disconnected
     *
     *  @param dev device
     *  @param intf interface number
     *  @param tptr pointer to the object to call the member function on
     *  @param mptr pointer to the member function to be called
     */
    template<typename T>
    void registerDriver(USBDeviceConnected * dev, uint8_t intf, T* tptr, void (T::*mptr)(void)) {
    }

    int BulkRead(USBEndpoint*ep, uint8_t* data, int size, int timeout_ms = -1);
    int IsochronousRead(USBEndpoint*ep, uint8_t* data, int size);

private:
    USBHost();
    static USBHost* inst;
    virtual bool addDevice(int hub, int port, bool lowSpeed);
    void root_enumeration(USBDeviceConnected* dev);
    void parseConfDescr(USBDeviceConnected* dev, uint8_t* conf_descr, uint32_t len, IUSBEnumerator* pEnumerator);
    USBDeviceConnected* DeviceLists[MAX_DEVICE_CONNECTED];
    int DeviceLists_count;

    int ControlRead(USBDeviceConnected* dev, SETUP_PACKET* setup, uint8_t* data, int size);
    int ControlWrite(USBDeviceConnected* dev, SETUP_PACKET* setup, uint8_t* data = NULL, int size = 0);
    int BulkWrite(USBEndpoint*ep, const uint8_t* data, int size);
    int InterruptRead(USBEndpoint*ep, uint8_t* data, int size);

    // USB HUB
    bool Hub(USBDeviceConnected* dev);
    int SetPortPower(USBDeviceConnected* dev, int port);
    int ClearPortPower(USBDeviceConnected* dev, int port);
    int PortReset(USBDeviceConnected* dev, int port);
    int SetPortFeature(USBDeviceConnected* dev, int feature, int index);
    int ClearPortFeature(USBDeviceConnected* dev, int feature, int index);
    int SetPortReset(USBDeviceConnected* dev, int port);
    int GetPortStatus(USBDeviceConnected* dev, int port, uint32_t* status);
};

