/* SCIboard(TM) SCIboard_ubloxGps.cpp
Copyright (c) 2013 K. Andres

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "SCIboard_ubloxGps.h"

#define SYNC_CHAR_1 0xB5
#define SYNC_CHAR_2 0x62

#define MAX_UBLOX_PACKET_SIZE  512

// UBX Class IDs
#define NAV_CLASS_ID    0x01    // Navigation
#define RXM_CLASS_ID    0x02    // Receiver
#define INF_CLASS_ID    0x04    // Information
#define ACK_CLASS_ID    0x05    // Ack/Nack
#define CFG_CLASS_ID    0x06    // Configuration input
#define MON_CLASS_ID    0x0A    // Monitoring messages
#define AID_CLASS_ID    0x0B    // AssistNow Aiding
#define TIM_CLASS_ID    0x0D    // Timing
#define ESF_CLASS_ID    0x10    // External Sensor Fusion


// Default - messages to keep
#define NAV_SOL         0x06    // TOWms, week, ECEF

// Turn ON some messages
#define NAV_SVINFO      0x30

// Turn off some default messages
#define NAV_POSLLH      0x02
#define NAV_STATUS      0x03
#define NAV_VELNED      0x12

#define TIM_TP          0x01
#define TIM_VRFY        0x06


extern Serial pc;
extern Serial gps;

void ublox_gps_callback();

#define UBLOX_BUF_SIZE  2048

// Input buffer
char ublox_msg_in[UBLOX_BUF_SIZE];
int nextin, nextout;

// Output buffer
char ublox_msg_out[UBLOX_BUF_SIZE];
char ublox_cnt;
char ublox_msg_out_flag;

char packetState;
int ublox_size;

#define UBLOX_MSG_CLASS     2       // field within msg
#define UBLOX_MSG_ID        3

void ubloxMsgRate(char msgclass, char id, char rate);
void ubloxBaudRate(void);

SCIboard_ubloxGps::SCIboard_ubloxGps() {
    ublox_cnt = 0;
    ublox_msg_out_flag = 0;
    nextin=nextout=0;
    packetState=0;
}

FILE *ublox_fp=NULL;

void SCIboard_ubloxGps::gps_startup(void) {
    gps.baud(38400);
    wait(.01);
    
    if(ublox_fp==NULL) {
        ublox_fp = fopen("/local/GPS.txt", "w");  // Open "GPS.txt" on the local file system for appending
    }

    // Turn off some of the default messages
    ubloxMsgRate(TIM_CLASS_ID, TIM_TP, 0);
    ubloxMsgRate(TIM_CLASS_ID, TIM_VRFY, 0);
    ubloxMsgRate(NAV_CLASS_ID, NAV_VELNED, 0);
    ubloxMsgRate(NAV_CLASS_ID, NAV_STATUS, 0);
    ubloxMsgRate(NAV_CLASS_ID, NAV_POSLLH, 0);

    // Turn on some messages
    ubloxMsgRate(NAV_CLASS_ID, NAV_SOL, 4);         // Change rate to once per second
    ubloxMsgRate(NAV_CLASS_ID, NAV_SVINFO, 8);      // Change rate to once per 2 seconds
    
    void ubloxBaudRate();    // Drop baud rate
    
    gps.attach(&ublox_gps_callback);
}

void SCIboard_ubloxGps::gps_close(void) {
    gps.attach(NULL);
    wait(0.01);
    
    if(nextin) {
        if(ublox_fp!=NULL) {
            memcpy(ublox_msg_out, ublox_msg_in, nextin);
            fwrite(ublox_msg_out, nextin, 1, ublox_fp);
        }  
        nextin=0;
    }

    if(ublox_fp!=NULL) {
        fclose(ublox_fp);
        ublox_fp=NULL;
    }
}


void ublox_gps_callback() {
    // Note: you need to actually read from the serial to clear the RX interrupt
    __disable_irq();
    ublox_msg_in[nextin++] = gps.getc();
    __enable_irq();

    if(nextin>=sizeof(ublox_msg_in)) {
        nextin=0;
        if(ublox_fp!=NULL) {
            memcpy(ublox_msg_out, ublox_msg_in, sizeof(ublox_msg_in));
            fwrite(ublox_msg_out, sizeof(ublox_msg_out), 1, ublox_fp);
        }  
    }
}


// Change message rate or turn off message
void ubloxMsgRate(char msgclass, char id, char rate) {
    unsigned char chkA=0, chkB=0;
    int n;
    uint8_t buf[] = {
        6,               // CLASS: CFG
        1,               // ID: MSG
        8,               // LENGTH
        0,               // LENGTH
        msgclass,
        id,
        0,
        rate,
        0, 0, 0, 0};
        
    gps.putc(SYNC_CHAR_1);
    gps.putc(SYNC_CHAR_2);
    
    for(n=0; n<sizeof(buf); n++) {
        gps.putc(buf[n]);
        chkA += buf[n];
        chkB += chkA;    
    }
    
    gps.putc(chkA);
    gps.putc(chkB);
}


// Change message baud rate --------------------------------
// Baud rate divisors
// 0, 0, 0x96, 0            // 38,400
// 0, 0, 0x4B, 0            // 19,200
// 0, 0x80, 0x25, 0         //  9,600
// 0, 0xC0, 0x12, 0         //  4,800

void ubloxBaudRate(void) {
    unsigned char chkA=0, chkB=0;
    int n;
    uint8_t buf[] = {
        6, 0,            // CLASS/id: CFG-PRT
        0x14, 0,         // LENGTH     
        01, 0,
        0, 0, 0xD0, 8, 0, 0, 0xC0, 0x12,
        0, 0, 7, 0, 1, 0, 0, 0,
        0, 0};

    gps.putc(SYNC_CHAR_1);
    gps.putc(SYNC_CHAR_2);
    
    for(n=0; n<sizeof(buf); n++) {
        gps.putc(buf[n]);
        chkA += buf[n];
        chkB += chkA;    
    }
    
    gps.putc(chkA);
    gps.putc(chkB);

    wait(.500);
    gps.baud(4800);
}
