/* SCIboard(TM) LSM303DLHC.cpp
Copyright (c) 2013 K. Andres

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "SCIboard_LSM303DLHC.h"

SCIboard_LSM303DLHC::SCIboard_LSM303DLHC(SCIboard_I2C *ptr_I2C) {
   pI2C = ptr_I2C;    
}

void SCIboard_LSM303DLHC::getDeviceID(unsigned char *id) {
    pI2C->readRegs(MAG_I2C_ADDR, MAGREG_IRA_REG, id, 3);
}

// ACCELEROMETER ------------------------------------------------------
void SCIboard_LSM303DLHC::setAccMode(unsigned char AccFs, unsigned char AccRate) {
    unsigned char data[2] = {ACCREG_CTRL_REG1, 0};
    float sf[] = {2, 4, 8, 16};
    
    accSF = 32768.0 / sf[AccFs];
    accRate = AccRate;
//    printf("AccRate=%u  AccFs=%u\r\n", AccRate, AccFs);
    
    data[1] = 0x07;                             // Enable accels
    data[1] |= AccRate << 4;
    pI2C->writeRegs(ACC_I2C_ADDR, data, 2);

    data[0] = ACCREG_CTRL_REG4;
    data[1] = AccFs << 4;
//    data[1] |= 0x80;                            // Block data update - BDU: 0=continuous
    pI2C->writeRegs(ACC_I2C_ADDR, data, 2);
}

unsigned char SCIboard_LSM303DLHC::getAccStatus(void) {
    unsigned char data;
    pI2C->readRegs(ACC_I2C_ADDR, ACCREG_STATUS, &data, 1);
    return data;
}

bool SCIboard_LSM303DLHC::bAccDataAvailable(void) {
    unsigned char data;
    pI2C->readRegs(ACC_I2C_ADDR, ACCREG_STATUS, &data, 1);
    if(data & STATUS_REG_ZYXDA) {
        return 1;
    }
    return 0;
}

void SCIboard_LSM303DLHC::getAccData(float *fData) {
    unsigned char data[6];

    pI2C->readRegs(ACC_I2C_ADDR, ACCREG_OUT_X_L, data, 6);
    
    fData[0] = float(short(data[1] << 8 | data[0])) / accSF;
    fData[1] = float(short(data[3] << 8 | data[2])) / accSF;
    fData[2] = float(short(data[5] << 8 | data[4])) / accSF;
}

unsigned char SCIboard_LSM303DLHC::getInt1Src(void) {
    unsigned char data;
    pI2C->readRegs(ACC_I2C_ADDR, ACCREG_INT1_SOURCE, &data, 1);
    return data;
}

void SCIboard_LSM303DLHC::writeCtrlReg(unsigned char reg, unsigned char value) {
    unsigned char data[2];
    
    if(reg>0 && reg<6) {
        data[0] = ACCREG_CTRL_REG1 - 1 + reg;
        data[1] = value;
        pI2C->writeRegs(ACC_I2C_ADDR, data, 2);
    }
}


// MAGNETOMETER ------------------------------------------------------
void SCIboard_LSM303DLHC::setMagMode(unsigned char MagFs, unsigned char MagRate) {
    unsigned char data[2] = {MAGREG_CRA_REG, 0};
    float GN[][2] = { 1100, 980, 
        855, 760,
        670, 600,
        450, 400,
        400, 355,
        330, 295,
        230, 205};
    
    magSF[0] = GN[MagFs-1][0];
    magSF[1] = magSF[0];
    magSF[2] = GN[MagFs-1][1];
//    printf("MagRate=%u  MagFs=%u\r\n", MagRate, MagFs);
//    printf("Mag SF= %.0f %.0f %.0f\r\n", magSF[0], magSF[1], magSF[2]);
    
    magRate = MagRate;
    data[1] = MagRate << 2;
//    data[1] |= 0x80;                            // Temperature sensor enable
    pI2C->writeRegs(MAG_I2C_ADDR, data, 2);

    data[0] = MAGREG_CRB_REG;
    data[1] = MagFs << 5;
    pI2C->writeRegs(MAG_I2C_ADDR, data, 2);

    data[0] = MAGREG_MR_REG;
    data[1] = MAG_Continuous_Conv;
    pI2C->writeRegs(MAG_I2C_ADDR, data, 2);
}

unsigned char SCIboard_LSM303DLHC::getMagStatus(void) {
    unsigned char data;
    pI2C->readRegs(MAG_I2C_ADDR, MAGREG_SR_REG, &data, 1);
    return data;
}

bool SCIboard_LSM303DLHC::bMagDataAvailable(void) {
    unsigned char data;
    pI2C->readRegs(MAG_I2C_ADDR, MAGREG_SR_REG, &data, 1);
    if(data & MAG_DRDY) {
        return true;
    }
    return false;
}

void SCIboard_LSM303DLHC::getMagData(float *fData) {
    char data[6];

    pI2C->readRegs(MAG_I2C_ADDR, MAGREG_OUT_X_H, (unsigned char*) data, 6);
    fData[0] = float(short(data[0] << 8 | data[1])) / 1100.0;
    fData[2] = float(short(data[2] << 8 | data[3])) / 980.0;
    fData[1] = float(short(data[4] << 8 | data[5])) / 1100.0;
}

void SCIboard_LSM303DLHC::getMagData(char *cData) {
    unsigned char data[6];

    pI2C->readRegs(MAG_I2C_ADDR, MAGREG_OUT_X_H, data, 6);
    memmove(cData, data, 6);    
}


float SCIboard_LSM303DLHC::getTemp(void) {
    char data[2];

    pI2C->readRegs(MAG_I2C_ADDR, MAGREG_TEMP_OUT_H, (unsigned char*) data, 2);
    return(float(short(data[0] << 8 | data[1]>>4)));
}

void SCIboard_LSM303DLHC::getTemp(char *cData) {
    char data[2];

    pI2C->readRegs(MAG_I2C_ADDR, MAGREG_TEMP_OUT_H, (unsigned char*) data, 2);
    memmove(cData, data, 2);
}
