/* SCIboard(TM) main.cpp
Copyright (c) 2013 K. Andres

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/* OVERVIEW
    Displays on USB serial (9600 bps) 
        MPL3115A2 ident 0xC4
        LSM303 ident 0x48 0x34 0x33
        L3G4200DTR ident 0xD3
    then collects 20 seconds of data from accelerometer and magnetometer
    
    Filename on Local: OUTxx.csv (csv format allows easy import into spreadsheets)
    Format: microseconds,record type, variable length data\r\n 
    Rec type 1= altimeter data altitude (meters) and semiconductor temperature (deg C)
    Rec type 2= accelerometer data x, y, and z in g's
    Rec type 3= magnetometer data x, y, and z
    Rec type 4= gyro data x, y, z in degrees per second (dps)
    Rec type 5= MOSFET switch events
    
    Items under development...
    MOSFET switch
    ublox GPS
*/

#include "mbed.h"
#include "math.h"

#include "SCIboard_I2C.h"
#include "SCIboard_MPL3115A2.h"
#include "SCIboard_LSM303DLHC.h"
#include "SCIboard_L3G4200DTR.h"

#include "SCIboard_DataLogger.h"
#include "SCIboard_ConfigFile.h"
#include "SCIboard_MOSFET.h"

#include "SCIboard_ubloxGps.h"

// LEDs
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);

// PWM out
PwmOut pwm1(p26);
PwmOut pwm2(p25);
PwmOut pwm3(p24);
PwmOut pwm4(p23);
PwmOut pwm5(p22);
PwmOut pwm6(p21);

Serial gps(p13, p14);
// Serial Xbee(p9, p10);

// Buzzer
DigitalOut alert(p29);      // CAN_RD

// MOSFET controls
DigitalOut fet_out1(p15);   // ADC1
DigitalOut fet_out2(p12);   // SPI2_MISO
DigitalOut fet_out3(p11);   // SPI2_MOSI
DigitalOut fet_out4(p8);    // GPIO8

// ADC inputs
AnalogIn batt_mon(p16);     // ADC2
AnalogIn fet_mon1(p17);     // ADC3
AnalogIn fet_mon2(p18);     // ADC4
AnalogIn fet_mon3(p19);     // ADC5
AnalogIn fet_mon4(p20);     // ADC6

//
Serial pc(USBTX, USBRX);    // Default 9600 bps

// Timers
Timer t;
int time_ms;


// I2C interface
//#define sdaPin p9
//#define sdcPin p10
#define sdaPin p28
#define sdcPin p27

SCIboard_I2C i2cBus(sdaPin, sdcPin);

// Altimeter
SCIboard_MPL3115A2 alt(&i2cBus);
InterruptIn MPL3115A2_INT1(p5);
void MPL3115A2_INT1_INTERRUPT(void);
//InterruptIn MPL3115A2_INT2();

const int accAltDecimate=5;
#define ALT_RATE    10
int accCnt=0;


// Accelerometer and magnetometer
SCIboard_LSM303DLHC lsm303(&i2cBus);
InterruptIn LSM303DLHC_DRDY(p7);
void LSM303DLHC_DRDY_INTERRUPT(void);
//InterruptIn LSM303DLHC_INT1(p6);
//void LSM303DLHC_INT1_INTERRUPT(void);
// INT2 n/c


// Gyro
SCIboard_L3G4200DTR l3g4200dtr(&i2cBus, GYR_DR_100HZ, GYR_BW_0, GYR_FS_250DPS);
//InterruptIn L3G4200DTR_INT();
//InterruptIn L3G4200DTR_DRDY(p24);
//void L3G4200DTR_INTERRUPT(void);


// Configuration data
void ReadConfigfile(void);
float fMachDelay, fMainDeploymentAlt, fLowVoltageAlarm, fAltLog, fAccelLog, fMagLog, fGyroLog, fUBLOXlog;

char *cfgKeyStr[]={
"MACH DELAY (s)",
"MAIN DEPLOYMENT (ft)",
"LOW VOLTAGE ALARM (v)",
"ALT LOG (1|0)",
"ACCEL LOG (1|0)",
"MAG LOG (1|0)",
"GYRO LOG (1|0)",
"UBLOX LOG (1|0)",
//"SSID",
//"PASSWD",
};

float *cfgFloatPtr[] = {
&fMachDelay, &fMainDeploymentAlt, &fLowVoltageAlarm, &fAltLog, &fAccelLog, &fMagLog, &fGyroLog, &fUBLOXlog};

// MOSFET switchs and ADC
SCIboard_Mosfet mosfet;

// UBLOX GPS
SCIboard_ubloxGps ubloxGps;

//---------------------------------------------------------------------
int main() {
    unsigned char data[10];
    float f[3];
    char str[132];
    int currentTime;
    int lastTime=0;
    float gyro_store[3];
    bool bGyroDataStored=false;

    // Ensure interrupts are off
    MPL3115A2_INT1.fall(NULL);
    LSM303DLHC_DRDY.fall(NULL);
//    LSM303DLHC_INT1.rise(NULL);
//    L3G4200DTR_DRDY.fall(NULL);
    
//    pc.baud(38400);

    ReadConfigfile();

    // DataLogger
    nextFilename();
    log_open();
    
    // Get I2C device IDs
    sprintf(str, "MPL3115A2 Ident 0x%X\r\n", alt.getDeviceID());
//    pc.printf(str);
    log_write(str);
    
    lsm303.getDeviceID(data);
    sprintf(str, "LSM303DLHC Ident 0x%X 0x%X 0x%X\r\n", data[0], data[1], data[2]);
//    pc.printf(str);
    log_write(str);
    
    sprintf(str, "L3G4200DTR Ident 0x%X\r\n", l3g4200dtr.getDeviceID());
//    pc.printf(str);
    log_write(str);
    
    // Setup accelerometer and magnetometer
    lsm303.setAccMode(ACC_4G, ACC_50HZ);
    lsm303.setMagMode(MAG_1p3G, MAG_75HZ);
    
    // Setup altimeter
    alt.setMode(ALT_MODE, OS4);
    alt.OST();  // initiate first conversion
    MPL3115A2_INT1.fall(MPL3115A2_INT1_INTERRUPT);
    
    if(fUBLOXlog) {
        ubloxGps.gps_startup();
    }
    
    t.start();  // start timer
    
    if(fMagLog) {
        LSM303DLHC_DRDY.fall(LSM303DLHC_DRDY_INTERRUPT);    // Magnetometer
    }
    
//    LSM303DLHC_INT1.rise(LSM303DLHC_INT1_INTERRUPT);    // Accelerometer


    // Collect some data then close file system to allow mbed to be visible on USB connected PC for demo app only
    while(t.read()<20) {
        time_ms = t.read_ms();
        currentTime = t.read();
        if(currentTime!=lastTime) {
            lastTime = currentTime;
            led2 = !led2;   // slow blink led
        }

        __disable_irq();
     
        // Accelerometer
        if(lsm303.bAccDataAvailable()) {
            lsm303.getAccData(f);
            if(fAccelLog) {
                sprintf(str,"%u,2,%.3f,%.3f,%.3f\r\n", time_ms, f[0], f[1], f[2]);
                log_write(str);
            }
            if(++accCnt >= accAltDecimate) {
                accCnt = 0;
                alt.OST();
            }            
        }


        #ifdef NON_DRDY_MAG  
        // Magnetometer      
        data[0] = lsm303.getMagStatus();
        pc.printf("MagStatus=%X\r\n", data[0]);

        if(lsm303.bMagDataAvailable()) {
            lsm303.getMagData(f);
            pc.printf("MAG: %.3f %.3f %.3f  %.0f\r\n", f[0], f[1], f[2], atan2(f[1],f[0])*180.0/PI);
        }
        #endif
        
        // Gyro
        if(fGyroLog) {
            if(l3g4200dtr.getStatus() & STATUS_REG_ZYXDA)
            {
                if(bGyroDataStored) {
                    l3g4200dtr.getData(f);
                    sprintf(str, "%u,4,%.3f,%.3f,%.3f\r\n", time_ms, f[0]+gyro_store[0], f[1]+gyro_store[1], f[2]+gyro_store[2]);
                    log_write(str);
                    bGyroDataStored = false;
                }
                else {
                    l3g4200dtr.getData(gyro_store);
                    bGyroDataStored = true;
                }
            }
        }

        __enable_irq();
    }

    // Ensure interrupts are off
    MPL3115A2_INT1.fall(NULL);
    LSM303DLHC_DRDY.fall(NULL);
//    LSM303DLHC_INT1.rise(NULL);
//    L3G4200DTR_DRDY.fall(NULL);

    if(fUBLOXlog) {
        ubloxGps.gps_close();
    }
    
    wait(.1);
    log_close();
    
    // Demo MOSFET
/*    pc.printf("Battery voltage=%f\r\n", mosfet.getBattVoltage());
    pc.printf("MOSFET voltage=%f\r\n", mosfet.getFetVoltage(4));
    mosfet.setFet(4, 2.0);
    pc.printf("MOSFET 4 on for 2 seconds.\r\n");
    pc.printf("MOSFET voltage=%f\r\n", mosfet.getFetVoltage(4));
    wait(3);
    pc.printf("MOSFET voltage=%f\r\n", mosfet.getFetVoltage(4));

    pc.printf("Done.\r\n");
    */
    led2=0;
        
    // Signal program done
    led1 = 1;
    led4 = led3 = led2 =0;
    while(1) {
        wait(0.2);
        data[0] = led4;
        led4 = led3;
        led3 = led2;
        led2 = led1;
        led1 = data[0];
    }
}


// Altimeter interrupt service ----------------------------------------
void MPL3115A2_INT1_INTERRUPT(void)
{
    float f[3];
    char str[80];
    
    __disable_irq();

    alt.getData(f);
    if(fAltLog) {
        sprintf(str, "%u,1,%.1f,%.1f\r\n", time_ms, f[0], f[1]);
        log_write(str);            
    }
   
    __enable_irq();
}


// Magnetometer interrupt service -------------------------------------
void LSM303DLHC_DRDY_INTERRUPT(void)
{
    float f[3];
    char str[80];
    
    __disable_irq();
    lsm303.getMagData(f);
    sprintf(str, "%u,3,%.3f,%.3f,%.3f\r\n", time_ms, f[0], f[1], f[2]);
    log_write(str);
    __enable_irq();
}


// Accelerometer interrupt service ------------------------------------
/*void LSM303DLHC_INT1_INTERRUPT(void)
{
    float f[3];
    unsigned char src;
    char str[80];
    
    __disable_irq();

    src = lsm303.getInt1Src();
    if(lsm303.bAccDataAvailable()) {
        lsm303.getAccData(f);
        sprintf(str,"%u,2,%.3f,%.3f,%.3f,%x\r\n", time_ms, f[0], f[1], f[2], src);
        log_write(str);
        if(++accCnt >= accAltDecimate) {
            accCnt = 0;
            alt.OST();
        }
    }
    
    __enable_irq();
}
*/


// Read configuration file --------------------------------------------
void ReadConfigfile(void) {
    ConfigFile cfgFile("/local/SCIbdCfg.txt");
    int n;
    float f;
    
    for(n=0; n<sizeof(cfgKeyStr)/sizeof(char*); n++) {
        if(cfgFile.getValue(cfgKeyStr[n], &f))
        {
            if(f < 0) f = 0;
            *(cfgFloatPtr[n]) = f;
            pc.printf("Found: %s: %f\r\n", cfgKeyStr[n], f);
        }
    }
    cfgFile.closeFile();
    
    if(fMachDelay > 15.0)
        fMachDelay = 15;
    
    if(fMainDeploymentAlt < 300)
        fMainDeploymentAlt = 300;
    else if(fMainDeploymentAlt > 2000)
        fMainDeploymentAlt = 2000;

    // save config file if not found?
   
}
