#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>

// Connect the Grove Shield to the UDK
// Connect the Grove Temperature Sensor to A0 on Shield
// http://www.seeedstudio.com/wiki/Grove_-_Temperature_Sensor
// Connect the Grove Light Sensor to the A1
// http://www.seeedstudio.com/wiki/Grove_-_Light_Sensor

AnalogIn in(PB_1); // This corresponds to A0 Connector on the Grove Shield
AnalogIn light(PB_0); // This corresponds to A1 Connector on the Grove Shield

float temperature;
float llevel;

// these options must match the settings on your Conduit
// uncomment the following lines and edit their values to match your configuration
static std::string config_network_name = "ATLANTA_TTN";
static std::string config_network_pass = "ATLANTA_TTN";
static uint8_t config_frequency_sub_band = 7;

float readTemperature(){
    float temp;
    int B=39751;
    float resistance;
    int a;
    // Read & Print Temperature
    a = in.read_u16();
    //printf("analog value: %i \r\n", a);
    resistance = (float)(65534-a)*10000/a; //get the resistance of the sensor;
    temp=1/(log(resistance/10000)/B+1/298.15)-273.15;//convert to temperature via datasheet
    //logInfo("Temperature: %f, temp);
    return temp;
}

float readLightSensor() {
    float sensorValue;
    float rsensor; 
    sensorValue = light.read();
    rsensor = (float)(1023-sensorValue)*10/sensorValue;
    printf("Sensor reading: %2.2f - %2.2f\r\n", sensorValue, rsensor);
  
    return rsensor;

}


int main() {
    int32_t ret;
    mDot* dot;
    std::vector<uint8_t> data;
    std::string data_str = "hello!";
    char string_buffer[64];
    std::string separator_str = ",";
    std::string temp_cls = "TC";
    
    // get a mDot handle
    dot = mDot::getInstance();
    
    // print library version information
    logInfo("version: %s", dot->getId().c_str());

    //*******************************************
    // configuration
    //*******************************************
    // reset to default config so we know what state we're in
    //dot->resetConfig();
    
    //dot->setLogLevel(mts::MTSLog::INFO_LEVEL);
    dot->setLogLevel(mts::MTSLog::TRACE_LEVEL);
    
    // set up the mDot with our network information: frequency sub band, network name, and network password
    // these can all be saved in NVM so they don't need to be set every time - see mDot::saveConfig()
    
    // frequency sub band is only applicable in the 915 (US) frequency band
    // if using a MultiTech Conduit gateway, use the same sub band as your Conduit (1-8) - the mDot will use the 8 channels in that sub band
    // if using a gateway that supports all 64 channels, use sub band 0 - the mDot will use all 64 channels
    
    logInfo("setting to public network");
    if ((ret = dot->setPublicNetwork(true)) != mDot::MDOT_OK) {
        logError("failed to set public network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }

    logInfo("setting frequency sub band");    
    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        logError("failed to set frequency sub band %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    logInfo("setting network name");
    if ((ret = dot->setNetworkName(config_network_name)) != mDot::MDOT_OK) {
        logError("failed to set network name %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    logInfo("setting network password");
    if ((ret = dot->setNetworkPassphrase(config_network_pass)) != mDot::MDOT_OK) {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    // in the 868 (EU) frequency band, spreading factors 7 - 12 are available
    logInfo("setting TX spreading factor");
    if ((ret = dot->setTxDataRate(mDot::SF_10)) != mDot::MDOT_OK) {
        logError("failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // request receive confirmation of packets from the gateway
    logInfo("enabling ACKs");
    if ((ret = dot->setAck(1)) != mDot::MDOT_OK) {
        logError("failed to enable ACKs %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // save this configuration to the mDot's NVM
    logInfo("saving config");
    if (! dot->saveConfig()) {
        logError("failed to save configuration");
    }
    //*******************************************
    // end of configuration
    //*******************************************

    // attempt to join the network
    logInfo("joining network");
    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        osDelay(std::max((uint32_t)1000, (uint32_t)dot->getNextTxMs()));
    }

    // format data for sending to the gateway
    //for (std::string::iterator it = data_str.begin(); it != data_str.end(); it++)
    //    data.push_back((uint8_t) *it);

    while (true) {
        data.clear();
        temperature = readTemperature();
        
        // Temperature
        sprintf(string_buffer, "%s%3.2f", "TC:", temperature);
        //sprintf(string_buffer, "%3.1f", temperature);
        for (int i = 0; i<strlen(string_buffer); i++)
        {
            data.push_back(((char*)string_buffer)[i]);
        }
            
        logDebug("Sending LoRa message, length: %d", data.size());
        logDebug("sending data: ");
        for(int i = 0; i < data.size(); i++)
        {
            printf("%c", data[i]);
        }
        printf("\n");
        
        // send the data to the gateway
        if ((ret = dot->send(data)) != mDot::MDOT_OK) {
            logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            logInfo("successfully sent data to gateway");
        }

        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        osDelay(std::max((uint32_t)5000, (uint32_t)dot->getNextTxMs()));
        
        data.clear();
        // Light Level
        llevel = readLightSensor();
        sprintf(string_buffer, "%s%5.1f", "LL:", llevel);
        for (int i = 0; i<strlen(string_buffer); i++)
        {
            data.push_back(((char*)string_buffer)[i]);
        }
        logDebug("Sending LoRa message, length: %d", data.size());
        logDebug("sending data: ");
        for(int i = 0; i < data.size(); i++)
        {
            printf("%c", data[i]);
        }
        printf("\n");
        
        // send the data to the gateway
        if ((ret = dot->send(data)) != mDot::MDOT_OK) {
            logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            logInfo("successfully sent data to gateway");
        }

        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        osDelay(std::max((uint32_t)5000, (uint32_t)dot->getNextTxMs()));
        
    }

    return 0;
}
