/*
 * (C) Copyright 2015 Valentin Ivanov. All rights reserved.
 *
 * This file is part of the RetroPlatform Library
 *
 * The RetroPlatform Library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>
 *
 * This library is inspired by Gamebuino Library (http://gamebuino.com)
 * from Aurélien Rodot. 
 */
#include "Retro.h"
#include "us_ticker_api.h"

const uint16_t OKSequence[]  =           {0x0005,0x138,0x168,0x0000};
const uint16_t CancelSequence[]  =       {0x0005,0x168,0x138,0x0000};
const uint16_t TickSequence[]  =               {0x0045,0x168,0x0000};

DigitalIn Retro::pin[NUM_BTN] = {
    DigitalIn(P0_14, PullUp), //left
    DigitalIn(P0_11, PullUp), //right
    DigitalIn(P0_12, PullUp), //down
    DigitalIn(P0_13, PullUp), //up
    DigitalIn(P0_16, PullUp), //robot
    DigitalIn(P0_1, PullUp)   //ship
};

Retro::Retro(): display(
        P0_19,
        P0_20,
        P0_7,
        P0_21,
        P0_22,
        P1_15,
        P0_2,
        LCD_ST7735::RGB), leftEye(P0_9,false), rightEye(P0_8,false)
{
    initialize();
}


void Retro::initialize()
{
    timePerFrame = 50000;
    frameStartUs = us_ticker_read();

    display.setOrientation(LCD_ST7735::Rotate270, false);
    display.setForegroundColor(Color565::White);
    display.setBackgroundColor(Color565::Black);

    sound.initialize();
    sound.setVolume(1);

    setFrameRate(20);
}

void Retro::setFrameRate(uint8_t fps)
{
    timePerFrame = 1000000 / fps;
    sound.prescaler = fps / 20;
    sound.prescaler = max(1, sound.prescaler);
}

bool once = false;

bool Retro::update()
{
    uint32_t current_time = us_ticker_read();
    frameDurationUs = current_time - frameStartUs;

    if( frameDurationUs > timePerFrame ) {
        frameStartUs = current_time;


        once = false;

        readButtons();
        return true;
    } else {
        if( !once ) {
            once = true;
            sound.update();

        }
        return false;
    }
}

void Retro::playOK()
{
    sound.playSequence(OKSequence,0);
}

void Retro::playCancel()
{
    sound.playSequence(CancelSequence,0);
}

void Retro::playTick()
{
    sound.playSequence(TickSequence,0);
}

bool Retro::collideCheck( int x1, int y1, int w1, int h1, int x2, int y2, int w2, int h2)
{
    return !( x2 >=  x1+w1  || x1 >= x2+w2  || y2 >=  y1+h1  || y1 >= y2 + h2 );
}

//Buttons
void Retro::readButtons()
{
    for (uint8_t thisButton = 0; thisButton < NUM_BTN; thisButton++) {
        if ( pin[thisButton].read() == 0 ) {
            _state[thisButton]++;
        } else {
            if (_state[thisButton] == 0)
                continue;
            if (_state[thisButton] == 0xFF)
                _state[thisButton] = 0;
            else
                _state[thisButton] = 0xFF;
        }
    }
}

bool Retro::pressed(uint8_t button)
{
    if (_state[button] == 1)
        return true;
    else
        return false;
}

bool Retro::released(uint8_t button)
{
    if (_state[button] == 0xFF)
        return true;
    else
        return false;
}

bool Retro::held(uint8_t button, uint8_t time)
{
    if(_state[button] == (time+1))
        return true;
    else
        return false;
}

bool Retro::repeat(uint8_t button, uint8_t period)
{
    if (period <= 1) {
        if ((_state[button] != 0xFF) && (_state[button]))
            return true;
    } else {
        if ((_state[button] != 0xFF) && ((_state[button] % period) == 1))
            return true;
    }
    return false;
}

uint8_t Retro::timeHeld(uint8_t button)
{
    if(_state[button] != 0xFF)
        return _state[button];
    else
        return 0;

}