/**
@file GameScreen.h
@brief Simple library for managing "Let The Ball Drop" Game screen
@brief extends on N5110 library created by Craig.A.Evans
@brief Adds functionality to animate game platforms and ball,
@brief also displays various game screens throughout gameplay.
@brief printString function from base class N5110 is overwritten and
@brief extened to vertical orientation text and inverted pixel scheme.
@author Thomas Davies
@date May 2015
*/


#ifndef GameScreen_H
#define GameScreen_H

#include "mbed.h"
#include "N5110.h"

/**
Platform structure
*/
struct Platform {
    int id;     ///< id identifier of platform
    int x;      ///< x-location of gap in platform
    int y;      ///< y-location of top pixel in platform
};

/**
Ball structure
*/
struct Ball {
    int x;      ///< ball x-location (far right pixel)
    int y;      ///< ball y-location (top pixel)
};

/** GameScreen Class
*
*Nokia 5110 screen controller with specific functions for manageing game objects and screens.
*
*/
class GameScreen: public N5110::N5110
{
public:

    /** Create GameScreen object connected to specific pins
    *
    * constructor inhereted from base class
    * @param pwr Pin connected to Vcc on the LCD display (pin 1)
    * @param sce Pin connected to chip enable (pin 3)
    * @param rst Pin connected to reset (pin 4)
    * @param dc  Pin connected to data/command select (pin 5)
    * @param mosi Pin connected to data input (MOSI) (pin 6)
    * @param sclk Pin connected to serial clock (SCLK) (pin 7)
    * @param led Pin connected to LED backlight (must be PWM) (pin 8)
    *
    */
    explicit GameScreen(PinName pwrPin, PinName scePin, PinName rstPin, PinName dcPin, PinName mosiPin, PinName sclkPin, PinName ledPin)
        :N5110(pwrPin, scePin,rstPin,dcPin,mosiPin,sclkPin,ledPin) {}

    /** Initialise Game Screen
    *
    *   Calls N5110::init() which powers up display and sets starting brightness
    *   initialises ball position and creates platform list.
    *
    */
    void Initialize();

    /** Print String
    *
    * Overrides N5110::printString(*) to allow vertical orientation text.
    * requires a refresh to display.
    *
    *@param str - string to be printed
    *@param x - x coordinate of the string
    *@param y - y coordinate of the string
    *@param invert - color inversion toggle (default = false)
    *
    */
    void printString(const char * str,int x,int y,bool invert = false);

    /** Draw Ball
    *
    * Draws ball on screen from playerBall struct in object, requires refresh.
    *
    */
    void drawBall();

    /** Erases Ball
    *
    * Erases ball on screen from playerBall struct in object, requires refresh.
    *
    */
    void eraseBall();

    /** Draws All Platforms
    *
    * Draws all platforms from array allPlatforms in object, requires refresh.
    *
    */
    void drawAllPlatforms();

    /** Erases All Platforms
    *
    * Erases all platforms on screen from array allPlatforms in object, requires refresh.
    *
    */
    void eraseAllPlatforms();

    /** Shifts Platforms Up
    *
    * Alter array of platforms to shift each platform up n pisxels, requires refresh.
    *@param n - number of pixels to shift platform.
    *
    */
    void shiftAllPlatforms(int n=1);

    /** Finds next closest platform
    *
    *@param y - vertical position on screen
    *@return next closest platform to y.
    *
    */
    Platform nextClosestPlatform(int y);

    /** Displays The Start Screen
    *
    * first screen user sees, author name & title
    *
    */
    void displayStartScreen();

    /** Displays Instruction Screen
    *
    * shows game instructions, navigation method and "back story"
    *
    */
    void displayInstructionScreen();

    /** Displays Countdown
    *
    * countdown from 5 to 1 displayed on screen.
    *
    */
    void displayCountdown();

    /** displays Game Over Screen
    *
    * shows player level, and displays options.
    * options can be highlighted according to cursor location array
    * @param lvl - game level to be displayed.
    * @param cursor - array of ints used to highlight options.
    * @param isRecord - only display record option if player achieved a record
    *
    */
    void displayEndScreen(int lvl,int *cursor,bool isRecord);

    /** Display Initial Entry Screen
    *
    * screen for user to enter initials.
    *
    *@param cursor - array of ints used to highlight letters
    *@param ltr1 - first letter of initials
    *@param ltr2 - second letter of initials
    *@param ltr3 - third letter of initials
    *
    */
    void displayRecordScreen(int *cursor, char ltr1, char ltr2,char ltr3);

    /** Display High Score Screen
    *
    *@param s_hs0 - initials of top player.
    *@param s_hs1 - initials of second player.
    *@param s_hs2 - initials of third player.
    *@param d_hs - array of high score levels.
    */
    void displayHighScores(char *s_hs0,char *s_hs1,char *s_hs2,int *d_hs);

    /** Display Second instruction Screen
    *
    */
    void displayInstructionScreen2();

    /** Display Pause Menu Screen
    *
    *@param cursor - array of ints used to highlight selection
    *@param isSound - toggle for "yes" or "no" text for sound
    *
    */
    void displayPauseScreen(int *cursor, char isSound);

    /** Sets Ball Position
    *
    *@param x - new x coordinate of ball (right side)
    *@param y - new y coordinate of ball (top)
    */
    void        setBallPos(int x, int y)        {
        playerBall.x = x;
        playerBall.y = y;
    }

    //ACCESSOR FUNCTIONS
    /** @return platform gap size */
    int         getPlatGapSize()                {
        return platGapSize_;
    }

    /** @return screen max Y */
    int         getMaxY()                          {
        return maxY_;
    }

    /** @return screen max X */
    int         getMaxX()                          {
        return maxX_;
    }

    /** @return platform thickness */
    int         getPlatThickness()                 {
        return platThickness_;
    }

    /** @return ball radius */
    int         getBallR()                      {
        return ballRadius_;
    }

    /** @return ball X coordinate */
    int         getBallX()                      {
        return playerBall.x;
    }

    /** @return ball Y coordinate*/
    int         getBallY()                      {
        return playerBall.y;
    }


    /** Creates All Platforms
    *
    * Randomly selects location of gap in each platform and adds all to 'allPlatforms' array.
    *
    */
    void createAllPlatforms();

    /** Draws Platform
    *
    * Draws a platform with thickness _platThickness at height y and gap x.
    *
    *@param x - location of gap on platform (right hand pixel)
    *@param y - y position of platform
    *
    */
    void drawPlatform(int x,int y);

    /** Draws Platform
    *
    * Erases a platform with thickness _platThickness at height y.
    *
    *@param y - y position of platform
    *
    */
    void erasePlatform(int y);

    /** Free All Platforms
    *
    * frees memory of each platform in 'allPlatforms' array and destroys the array.
    *
    */
    void freeAllPlatforms();

    static const int platGapSize_ = 8;          ///< standard platform gap width in pixels
    static const int platThickness_ = 2;        ///< platform thickness in pixels
    static const int platSpacing_ = 14;         ///< subsequent platform spacing in pixels
    static const int maxX_ = 48;                ///< maximun horizontal pixel
    static const int maxY_ = 84;                ///< maximum vertical pixel
    static const int ballRadius_ = 4;           ///< size of player ball
    static const int numPlatforms_ = 6;         ///< total number of platforms
    Ball playerBall;                            ///< player ball structure
    Platform *allPlatforms[numPlatforms_];      ///< array used to track each platform, and refresh when needed!
};


#endif