/**
@file main.cpp
@brief Source file for main game containing function defintions and main loop.
@autor Thomas Davies
@date May 2015
*/

#include "main.h"


int main()
{
    DigitalOut backLight(p26);
    backLight.write(1);
    lcd.Initialize();
    powerSave();

    lcd.displayStartScreen();
    while(!isRight()) {}       //wait until right button pressed

    wait(0.5);          //just to ignore layover from first button press.

    lcd.displayInstructionScreen();
    while(!isRight()) {}    //wait until right button pressed.

    wait(0.5);
    lcd.displayInstructionScreen2();
    while(!isRight()) {}    //wait until right button pressed.

    lcd.displayCountdown();
    lcd.clear();

    bool refresh = false;

    lcd.drawAllPlatforms();
    lcd.drawBall();
    lcd.refresh();

    gameClock.attach(&clockCounter,0.003);  //attach the master clock!

    //MAIN GAME LOOP
    //multiple runs on a single loop. stop with a bool

    char buffer[10];
    while(true) {
        if (isFirstCheck) { //if first time checking this clock cycle.
            if(clockCount %(gameSpeed) == 0) {
                //advance platforms
                advancePlatforms(numPixelsToJump);
                numPlatformShifts ++;

                if (numPlatformShifts%5 == 0) {
                    gameLevel ++;
                }
                refresh = true;
            }

            if(clockCount%14 == 0) {
                //ball movement
                if (isLeft() && isBallDirClear(1)) {    //if josystick left and direction is clear
                    lcd.eraseBall();
                    lcd.setBallPos(lcd.getBallX()+2,lcd.getBallY());
                    refresh = true;

                } else if (isRight() && isBallDirClear(0)) {
                    lcd.eraseBall();
                    lcd.setBallPos(lcd.getBallX()-2,lcd.getBallY());
                    refresh = true;
                }
            }

            //if the ball is in free space, animate it falling!
            if (clockCount%(10/numPixelsToJump) == 0) {
                if(isBallFalling(lcd.getBallY())) {
                    lcd.eraseBall();
                    lcd.setBallPos(lcd.getBallX(),lcd.getBallY() + numPixelsToJump);
                    refresh = true;
                    isFirstHit = true;
                }
            }

            if ((clockCount%(-45*gameSpeed + 2045) == 0)) {    //starting levels are quick to progress, later slower.
                if (isFirstSpeedUp) //skip first accepted modulus!
                    isFirstSpeedUp = false;
                else {
                    gameSpeed --;
                    isFirstSpeedUp = true;
                }
            }

            if (joystickButton.read()) {    //pause menu!
                int oldClock = clockCount;  //save the clock count!
                pauseScreenController();
                clockCount = oldClock;
                wait(0.3);
                lcd.clear();
                refresh = true;
            }

            if (refresh) {
                lcd.drawBall();
                lcd.drawAllPlatforms();

                sprintf(buffer,"lvl:%d",gameLevel);
                lcd.printString(buffer,lcd.getMaxX() - 2,-1);
                lcd.refresh();
                refresh = false;
            }


            //check if ball hit roof
            if (lcd.getBallY() == 7) {
                break;
            }


            isFirstCheck = false;
        }
    }

    gameClock.detach();

    lcd.clear();

    endScreenController();
    return 1;
}


//#############FUNCTION DEFINITIONS#################//
void advancePlatforms(int n)
{
    //ADVANCE
    lcd.eraseAllPlatforms();
    lcd.shiftAllPlatforms(n);
    lcd.drawAllPlatforms();

    //if ball is on platform
    if(!isBallFalling(lcd.getBallY()-n)) {
        //auditory feedback if first collision
        if (isFirstHit) {
            boop();
            isFirstHit = false;
        }
        //move ball up with platform
        lcd.eraseBall();
        lcd.setBallPos(lcd.getBallX(),lcd.getBallY() -n);
    }


}

bool isBallFalling(int bY)
{
    int bX = lcd.getBallX();    //ball X pos
    int bR = lcd.getBallR();    //ball radius

    //find next platform lower than ball
    //check Y pos to see if ball is on platform, return false if on plat
    //check X pos to see if ball is over gap, return true if over gap
    Platform closestPlatform = lcd.nextClosestPlatform(bY - bR);


    //if bottom of ball on level of platform
    if ((bY+bR) == (closestPlatform.y)) {
        if (bX > closestPlatform.x && (bX+bR-1) < (closestPlatform.x + lcd.getPlatGapSize())) { //OVER GAP
            return true;
        } else {
            return false;
        }      //On platform!
    } else if ((bY+bR) >= (lcd.getMaxY()-1)) {
        return false;
    } else  //FREE SPACE
        return true;
}


bool isBallDirClear(int dir)
{
    Platform closestPlatform = lcd.nextClosestPlatform(lcd.getBallY() - lcd.getBallR()+2);

    int ballRight = lcd.getBallX();
    int ballLeft = ballRight+lcd.getBallR();
    int ballTop = lcd.getBallY();
    int ballBottom = ballTop + lcd.getBallR();

    if (dir == 1) {
        if (ballBottom > closestPlatform.y)    //is ball inside a gap in platform?
            if (ballLeft >= (closestPlatform.x+lcd.getPlatGapSize()))    //is the ball next to gap wall?
                return false;
        if (ballLeft < lcd.getMaxX()) {    //is ball next to left hand wall?
            return true;
        }
    } else if (dir == 0) {
        if (ballBottom > closestPlatform.y)    //is ball inside a gap in platform?
            if (ballRight <= closestPlatform.x)
                return false;

        if (ballRight > 0) {   //right hand wall check
            return true;
        }
    }
    return false;
}

bool isRight()
{
    double count = 0.0;
    double total = 0.0;
    //obtain an average to negate outliers
    while (count < 10) {
        count ++;
        total += joystickX.read();
    }
    if ((total/count)> 0.8)
        return true;

    return false;
}

bool isLeft()
{
    double count = 0.0;
    double total = 0.0;
    //obtain an average to negate outliers
    while (count < 10) {
        count ++;
        total += joystickX.read();
    }
    if ((total/count) <  0.2)
        return true;

    return false;
}

bool isUp()
{
    double count = 0.0;
    double total = 0.0;
    //obtain an average to negate outliers
    while (count < 10) {
        count ++;
        total += joystickY.read();
    }

    if ((total/count) <  0.2)
        return true;

    return false;
}
bool isDown()
{
    double count = 0.0;
    double total = 0.0;
    //obtain an average to negate outliers
    while (count < 10) {
        count ++;
        total += joystickY.read();
    }
    if ((total/count) >  0.8)
        return true;

    return false;
}

void clockCounter ()
{
    //increment clock count!
    clockCount ++;
    isFirstCheck = true;
}

void endScreenController()
{
    int cursorLoc = 0;
    //cursor location array.
    int cursorArray[3][3] = {
        {1,0,0},
        {0,1,0},
        {0,0,1}
    };

    //check if user achieved high score
    bool record = isRecord();

    //if not a record move mouse start to second option
    if (!record)
        cursorLoc = 1;

    lcd.displayEndScreen(gameLevel,cursorArray[cursorLoc],record);
    bool change = false;

    while (true) {
        if (isUp()) {
            cursorLoc ++;   //advance cursor menu
            change = true;
        } else if (isDown()) {
            cursorLoc --;   //move back cursor menu
            change = true;
        }

        //cyclical cursor management
        if (cursorLoc == 3)
            cursorLoc = 1-record;
        else if (cursorLoc == (-record))
            cursorLoc = 2;

        //if theres a change to buffer, show it
        if (change) {
            lcd.displayEndScreen(gameLevel,cursorArray[cursorLoc],record);
            wait(0.5);
            change = false;
        }

        //menu selection manager
        if (joystickButton.read()) {
            switch(cursorLoc) {
                case 0:
                    //take to initial entry page! (only allow if score > top 3)
                    wait(0.3);
                    highScoreEditor(recordEntryController(),record);

                    wait(1);
                    //reset once button pressed!
                    while(!joystickButton.read()) {}
                    mbed_reset();
                    break;
                case 1:
                    //take to high score page!
                    highScoreEditor("",0);
                    wait(1);
                    //reset once button pressed.
                    while(!joystickButton.read()) {}
                    mbed_reset();
                    break;
                case 2:
                    //restart
                    mbed_reset();
                    break;
            }
        }
    }
}

char* recordEntryController()
{
    //array of letters to traverse
    char *letters = {"ABCDEFGHIJKLMNOPQRSTUVWXYZ"};

    //array of ints, each corresponding to letter in letters array
    int ptr[3] = {0,0,0};

    //cursor management variables
    int cursorLoc = 0;
    int cursorArray[3][3] = {
        {1,0,0},
        {0,1,0},
        {0,0,1}
    };

    lcd.displayRecordScreen(cursorArray[cursorLoc],letters[ptr[0]],letters[ptr[1]],letters[ptr[2]]);

    bool change = false;

    while (true) {
        if (isUp()) {
            cursorLoc ++;
            change = true;
        } else if (isDown()) {
            cursorLoc --;
            change = true;
        } else if (isLeft()) {
            //cyclical letter management
            if (ptr[cursorLoc] == 0)
                ptr[cursorLoc] = 25;   //last letter
            else
                ptr[cursorLoc] --;  //previous letter
            change = true;
        } else if (isRight()) {
            if (ptr[cursorLoc] == 25)
                ptr[cursorLoc] = 0;  //first letter
            else
                ptr[cursorLoc] ++;  //next letter
            change = true;
        }

        //cyclical mouse management
        if (cursorLoc == 3)
            cursorLoc = 0;
        else if (cursorLoc == -1)
            cursorLoc = 2;

        if (change) {
            lcd.displayRecordScreen(cursorArray[cursorLoc],letters[ptr[0]],letters[ptr[1]],letters[ptr[2]]);
            wait(0.2);
            change = false;
        }

        if (joystickButton.read()) {
            //go to high score screen
            static char initials[10];
            sprintf(initials,"%c%c%c",letters[ptr[0]],letters[ptr[1]],letters[ptr[2]]);
            return initials;
        }
    }
}

void highScoreEditor(char *playerInitials,bool isRecord)
{
    //check if lvl is in top 3
    FILE *fp = fopen("/local/scores.csv","r");  //open for reading.

    char s_hs0[3],s_hs1[3],s_hs2[3];
    int d_hs[3];

    //if file exists, parse it
    if (fp != NULL) {

        char buffer[2];

        fscanf(fp,"%3s,%s",s_hs0,buffer);
        d_hs[0] = atoi(buffer);
        fscanf(fp,"%3s,%s",s_hs1,buffer);
        d_hs[1] = atoi(buffer);
        fscanf(fp,"%3s,%s",s_hs2,buffer);
        d_hs[2] = atoi(buffer);

        fclose(fp);
    }
    //else set arbitrary values!
    else {
        d_hs[0] = 0;
        d_hs[1] = 0;
        d_hs[2] = 0;

        strcpy(s_hs0,"---");
        strcpy(s_hs1,"---");
        strcpy(s_hs2,"---");
    }

    if (isRecord) {

        if (gameLevel >= d_hs[2]) {
            d_hs[2] = gameLevel;
            strncpy(s_hs2,playerInitials,3);
        }
        if (gameLevel >= d_hs[1]) {
            d_hs[2] = d_hs[1];
            d_hs[1] = gameLevel;
            strncpy(s_hs2,s_hs1,3);
            strncpy(s_hs1,playerInitials,3);
        }
        if (gameLevel >= d_hs[0]) {
            d_hs[1] = d_hs[0];
            d_hs[0] = gameLevel;
            strncpy(s_hs1,s_hs0,3);
            strncpy(s_hs0,playerInitials,3);
        }

        char buffer[40];
        sprintf(buffer,"%s,%d\n%s,%d\n%s,%d",s_hs0,d_hs[0],s_hs1,d_hs[1],s_hs2,d_hs[2]);
        fp = fopen("/local/scores.csv","w");
        fprintf(fp,buffer);
        fclose(fp);
    }

    lcd.displayHighScores(s_hs0,s_hs1,s_hs2,d_hs);
}

bool isRecord ()
{
    //check if lvl is in top 3
    FILE *fp = fopen("/local/scores.csv","r");  //open for reading.

    //if file doesn't exist, then player has a highscore!
    if (fp == NULL)
        return true;

    int highScore;
    char buffer[2];
    char temp[3];

    for (int i = 0; i < 3; i++) {
        fscanf(fp,"%3s,%s",temp,buffer);
        highScore = atoi(buffer);
        if (gameLevel > highScore) {
            fclose (fp);
            return 1;
        }
    }
    fclose(fp);
    return 0;
}

void boop ()
{
    if (isSound == 'Y') {
        buzzer.write(1);
        lowFlipper.attach_us(&pwmLow,pwmCount);
    }
}

void pwmLow()
{
    if (pwmCount != 4000) {
        buzzer.write(0);
        highFlipper.attach_us(&boop,pwmCount);
        pwmCount += 50;
    } else
        pwmCount = 1000;

}

//can toggle sound or quit game.
void pauseScreenController()
{
    int cursorLoc = 0;
    int cursorArray[3][3] = {
        {1,0,0},
        {0,1,0},
        {0,0,1}
    };

    lcd.displayPauseScreen(cursorArray[cursorLoc],isSound);

    bool change = false;
    bool resume = false;

    while (!resume) {
        if (isUp()) {
            cursorLoc ++;
            change = true;
        } else if (isDown()) {
            cursorLoc --;
            change = true;
        } else if (isLeft() || isRight()) {
            if (isSound == 'Y')
                isSound = 'N';
            else
                isSound = 'Y';
            change = true;
        }

        //cyclical mouse management
        if (cursorLoc == 3)
            cursorLoc = 0;
        else if (cursorLoc == -1)
            cursorLoc = 2;

        if (change) {
            lcd.displayPauseScreen(cursorArray[cursorLoc],isSound);
            wait(0.2);
            change = false;
        }

        if (joystickButton.read()) {
            switch (cursorLoc) {
                case 1:
                    mbed_reset();
                    break;
                case 2:
                    resume = true;
                    break;
                default:
                    break;
            }
        }
    }
}

void powerSave()
{
    PHY_PowerDown();    //power down ethernet
    //mbed automatically manages peripheral power, no need to manually disable each
}
