/**
@file main.h
@brief Header file for main game containing function prototypes, namespaces and global variables.
@autor Thomas Davies
@date May 2015
*/

#ifndef main_H
#define main_H

#include "mbed.h"
#include "GameScreen.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

/**
@namespace serial
@brief serial communication with host
*/
Serial serial(USBTX,USBRX);

/**
@namespace lcd
@brief game screen controller
*/
GameScreen lcd(p7,p8,p9,p10,p11,p13,p21);

/**
@namespace leds
@brief Output for onboard LEDs
*/
BusOut leds(LED4,LED3,LED2,LED1);

/**
@namespace joystickX
@brief input from joystick X, on p15
*/
AnalogIn joystickX(p15);

/**
@namespace joystickY
@brief input from joystick Y, on p16
*/
AnalogIn joystickY(p16);

/**
@namespace joystickButton
@brief input from joystick button, on p17
*/
DigitalIn joystickButton(p17);

/**
@namespace buzzer
@brief digital output to buzzer on p5
*/
DigitalOut buzzer(p5);

/**
@namespace local
@brief local file system
*/
LocalFileSystem local("local");

/**
@namespace gameClock
@brief the master clock of game operation. All animation timings are based on this clock.
*/
Ticker gameClock;

/**
@namespace lowFlipper
@brief used for PWM speaker control, triggers pin low.
*/
Timeout lowFlipper;

/**
@namespace highFlipper
@brief used for PWM speaker control, triggers pin high.
*/
Timeout highFlipper;

//##############GLOBAL VARIABLES##################//

int clockCount = 0;         ///< master clock counter
bool isFirstCheck = true;   ///< first check in clock cycle?
bool isFirstHit = true;     ///< first hit on a platform?
bool isFirstSpeedUp = true; ///< first game speed up?
int gameLevel = 0;          ///< current game level
int gameSpeed = 41;         ///< current game speed
int numPixelsToJump = 1;    ///< number of pixels ball/plat moves per animation
int numPlatformShifts = 0;  ///< number of times platforms shifted, used for score calculation
int pwmCount = 1000;        ///< pwm count for platform collision sound
char isSound = 'Y';         ///< sound toggle, 'Y' for sound enabled.

//#############FUNCTION PROTOTYPES##################//

/** Reset Mbed */
extern "C" void mbed_reset();

/** Advance Platforms
*
* animate platforms shift up n pixels
*
* @param n - number of pixels to shift
*/
void advancePlatforms(int n=1);

/** Check if ball falling
*
* Falling if bal is in free space, not on platform
* @param bY - ball Y coordinate
* @return 1 if falling 0 if not.
*/
bool isBallFalling(int bY);

/** Is Joystick Right?
*
* reads joystickX value and check direction
*@returns 1 if right 0 if not.
*/
bool isRight();

/** Is Joystick Left?
*
* reads joystickX value and check direction
*@returns 1 if left 0 if not.
*/
bool isLeft();

/** Is Joystick Up?
*
* reads joystickY value and check direction
*@returns 1 if up 0 if not.
*/
bool isUp();

/** Is Joystick Down?
*
* reads joystickY value and checks direction
*@returns 1 if down 0 if not.
*/
bool isDown();

/** ISR - Clock Counter
*
*increments number of clock cycles
*/
void clockCounter ();

/** Is Ball Direction Clear?
*
*given ball direction determines if path is clear to move.
*@param dir - direction ball is travelling in (0 for left 1 for right)
*@returns 1 if clear 0 if not
*/
bool isBallDirClear(int dir);

/** Game Over Controller
*
* Displays end game screen, and allows user interaction with options via joystick.
* Also controls option selection and next screens.
*/
void endScreenController();

/** Record Entry Controller
*
* Displays initial entry screen, allows user to scroll through letters and select initials.
* @return intitials entered by user.
*/
char* recordEntryController();

/** is Record?
*
* check if score achieved by player is a record
* @return 1 if gameLevel is a record 0 if not
*/
bool isRecord ();

/** High Score Editor
*
* Manages highscore file and recordEntryController().
*/
void highScoreEditor(char *playerInitials,bool isRecord);

/** Pause Screen Controller
*
* Displays pause screen and allows user interaction to toggle sound,restart, and resume game.
*/
void pauseScreenController();

/** PWM speaker control function
*
* makes a boop sound used for collisions with platforms.
*/
void boop ();

/** Boop Helper function
*
* sets pin low when called, then starts timer to recall Boop();
*/
void pwmLow();

/** Power Saving Mode
*
* Powers down unused peripherals & ethernet
*/
void powerSave();


#endif

