/*
    TLVList.h 
    Copyright (c) Donatien Garnier 2012
    donatien.garnier@appnearme.com
    http://www.appnearme.com/
*/

/** \file TLVList.h
*    List of Type/Length/Values items
*/

#ifndef TLVLIST_H_
#define TLVLIST_H_

#include <cstdint> //For uint_*t
#include <cstring> //For size_t

using std::uint8_t;
using std::uint16_t;
using std::uint32_t;
using std::size_t;

typedef struct __tlv_list tlv_list;

/** A simple container
* List of Type/Length/Values items
*/
class TLVList
{
public:

  //getNext...
  //get()
  //put()


  //typedef tlv_type TLVType;
  /** Type of a TLV item
   *
   */
  enum TLVType
  {
    UINT8, //!< Unsigned char (byte)
    UINT32, //!< Unsigned int
    UINT8_ARRAY, ///< Byte array
    STRING, //!<String
    NONE, //!<End of packet
    UNKNOWN //!< Unknown type
  };

  /* Getters */

  /** Iterate to next item.
   *  @return type if next item exists, NONE otherwise
   */
  TLVType getNext();

  /** Get current item's type.
   *  @return type if item exists, NONE otherwise
   */
  TLVType getType();

  /** Get uint8_t value.
   * If the current item's type is uint32_t, the value will be masked with 0xFF.
   *  @return uint8_t value OR 0 if the type is incompatible
   */
  uint8_t getUInt8();

  /** Get uint32_t value.
   * If the current item's type is uint8_t, the value will be casted to uint32_t.
   *  @return uint32_t value OR 0 if the type is incompatible
   */
  uint32_t getUInt32();

  /** Get array length.
   *  @return bytes array length
   */
  size_t getArrayLength();

  /** Get array.
   *  @param buf pointer to buffer's start
   *  @param maxLen maximum number of bytes to copy
   *  @return number of copied bytes
   */
  size_t getArray(uint8_t* buf, size_t maxLen);

  /** Get string length.
   *  @return string length
   */
  size_t getStringLength();

  /** Get string.
   *  Copy string to buffer (including null-terminating char).
   *  @param str pointer to string's start
   *  @param maxLen maximum number of chars to copy (not including null-terminating char)
   *  @return number of copied chars
   */
  size_t getString(char* str, size_t maxLen);

  /* Setters */

  /** Check whether there is space left in list.
   *  @return true if there is space left, false otherwise
   */
  bool isSpace();

  /** Put uint8_t value.
   *  @param value uint8_t value
   *  @return true on success, false if there is not enough space in buffer
   */
  bool putUInt8(uint8_t value);

  /** Put uint32_t value.
   *  @param value uint32_t value
   *  @return true on success, false if there is not enough space in buffer
   */
  bool putUInt32(uint32_t value);

  /** Put array.
   *  @param buf pointer to buffer's start
   *  @param len number of bytes to copy
   *  @return number of copied bytes
   */
  size_t putArray(uint8_t* buf, size_t len);

  /** Put string.
   *  @param str pointer to null-terminated string's start
   *  @return number of copied bytes
   */
  size_t putString(char* str);

protected:
  TLVList();

  void wrap(tlv_list* payload);

private:
  tlv_list* m_tlvList;

  TLVType m_type;

  union
  {
    uint8_t m_uint8;
    uint32_t m_uint32;
    uint8_t* m_array;
    char* m_str;
  };

  union
  {
    size_t m_arrayLen;
    size_t m_strLen;
  };

  bool m_space;

  friend class NdefCallback;
};


#endif /* TLVLIST_H_ */
