/*
    TLVList.cpp 
    Copyright (c) Donatien Garnier 2012
    donatien.garnier@appnearme.com
    http://www.appnearme.com/
*/

#include "TLVList.h"

#include "munfc/core/fwk.h"
#include "munfc/ndef/appnearme_ndef_tlv.h"

#include <cstring> //For memcpy, strlen
using std::memcpy;
using std::strlen;

TLVList::TLVList() : m_tlvList(NULL)
{

}

void TLVList::wrap(tlv_list* payload)
{
  m_tlvList = payload;
  m_space = true;
  m_type = NONE;
}



/* Getters */

/** Iterate to next item
 *  @return type if next item exists, NONE otherwise
 */
TLVList::TLVType TLVList::getNext()
{
  if(!m_tlvList)
  {
    return NONE;
  }

  m_type = (TLVList::TLVType) appnearme_ndef_tlv_next_type(m_tlvList); //Explicit cast, this is the same enum
  switch(m_type)
  {
  case UINT8:
    m_arrayLen = 0;
    m_uint8 = appnearme_ndef_tlv_get_uint8(m_tlvList);
    break;
  case UINT32:
    m_arrayLen = 0;
    m_uint32 = appnearme_ndef_tlv_get_uint32(m_tlvList);
    break;
  case UINT8_ARRAY:
    //In this order, otherwise would break state-machine
    m_arrayLen = appnearme_ndef_tlv_get_array_length(m_tlvList);
    m_array = appnearme_ndef_tlv_get_array(m_tlvList);
    break;
  case STRING:
    //In this order, otherwise would break state-machine
    m_strLen = appnearme_ndef_tlv_get_string_length(m_tlvList);
    m_str = appnearme_ndef_tlv_get_string(m_tlvList);
    break;
  }
  return m_type;
}

/** Get current item's type.
*  @return type if item exists, NONE otherwise
*/
TLVList::TLVType TLVList::getType()
{
  return m_type;
}

/** Get uint8_t value
 * If the current item's type is uint32_t, the value will be masked with 0xFF
 *  @return uint8_t value OR 0 if the type is incompatible
 */
uint8_t TLVList::getUInt8()
{
  switch(m_type)
  {
  case UINT8:
    return m_uint8;
  case UINT32:
    return m_uint32 & 0xFF;
  default:
    return 0;
  }
}

/** Get uint32_t value
 * If the current item's type is uint8_t, the value will be casted to uint32_t
 *  @return uint32_t value OR 0 if the type is incompatible
 */
uint32_t TLVList::getUInt32()
{
  switch(m_type)
  {
  case UINT32:
    return m_uint32;
  case UINT8:
    return m_uint8 & 0xFF;
  default:
    return 0;
  }
}

/** Get array length
 *  @return bytes array length
 */
size_t TLVList::getArrayLength()
{
  switch(m_type)
  {
  case UINT8_ARRAY:
    return m_arrayLen;
  default:
    return 0;
  }
}

/** Get array
 *  @param buf pointer to buffer's start
 *  @param maxLen maximum number of bytes to copy
 *  @return number of copied bytes
 */
size_t TLVList::getArray(uint8_t* buf, size_t maxLen)
{
  if(m_type != UINT8_ARRAY)
  {
    return 0;
  }
  size_t len = MIN(maxLen, m_arrayLen);
  memcpy(buf, m_array, len);
  return len;
}

/** Get string length
 *  @return string length
 */
size_t TLVList::getStringLength()
{
  switch(m_type)
  {
  case STRING:
    return m_strLen;
  default:
    return 0;
  }
}


/** Get string
 *  Copy string to buffer (including null-terminating char)
 *  @param str pointer to string's start
 *  @param maxLen maximum number of chars to copy (not including null-terminating char)
 *  @return number of copied chars
 */
size_t TLVList::getString(char* str, size_t maxLen)
{
  if(m_type != STRING)
  {
    return 0;
  }
  size_t len = MIN(maxLen, m_strLen);
  memcpy(str, m_str, len);
  str[len] = '\0'; //Add null-terminating char
  return len;
}

/* Setters */

/** Check whether there is space left in list
 *  @return true if there is space left, false otherwise
 */
bool TLVList::isSpace()
{
  return m_space;
}

/** Put uint8_t value
 *  @param value uint8_t value
 *  @return true on success, false if there is not enough space in buffer
 */
bool TLVList::putUInt8(uint8_t value)
{
  int res = appnearme_ndef_tlv_put_uint8(m_tlvList, value);
  if( !res )
  {
    m_space = false;
    return false;
  }
  return true;
}

/** Put uint32_t value
 *  @param value uint32_t value
 *  @return true on success, false if there is not enough space in buffer
 */
bool TLVList::putUInt32(uint32_t value)
{
  int res = appnearme_ndef_tlv_put_uint32(m_tlvList, value);
  if( !res )
  {
    m_space = false;
    return false;
  }
  return true;
}

/** Put array
 *  @param buf pointer to buffer's start
 *  @param len number of bytes to copy
 *  @return number of copied bytes
 */
size_t TLVList::putArray(uint8_t* buf, size_t len)
{
  int res = appnearme_ndef_tlv_put_array(m_tlvList, buf, len);
  if( !res )
  {
    m_space = false;
    return 0;
  }
  return len;
}

/** Put string
 *  @param str pointer to null-terminated string's start
 *  @return number of copied bytes
 */
size_t TLVList::putString(char* str)
{
  int res = appnearme_ndef_tlv_put_string(m_tlvList, str);
  if( !res )
  {
    m_space = false;
    return 0;
  }
  return strlen(str);
}




