/*
    MuNFC.h
    Copyright (c) Donatien Garnier 2012
    donatien.garnier@appnearme.com
    http://www.appnearme.com/
*/


#ifndef MUNFC_H_
#define MUNFC_H_

#include "MuNFCConfig.h"

#include <cstdint> //For uint_*t
#include <cstring> //For size_t

using std::uint8_t;
using std::uint16_t;
using std::uint32_t;
using std::size_t;

#include "NFCEvent.h"
#include "NdefCallback.h"
#include "EventCallback.h"
#include "TLVList.h"

#include "mbed.h"
#if MUNFC_RTOS
#include "rtos/rtos.h"
#endif

/** A library for embedded NFC applications using NXP's PN512/PN532 NFC transceivers.
* Visit http://www.appnearme.com/
*/
class MuNFC
{
public:


  /** Instantiate the NFC stack for the following mobile app and using the following PN512/PN532 chip.
   * @param appHash 16 chars-long hash of the corresponding mobile app
   * @param version Minimum version of the mobile app to use in BCD format encoded as an uint32_t (0x01000000 is version 1.0.0.0)
   * @param mosi MOSI pin of the SPI interface
   * @param miso MISO pin of the SPI interface
   * @param sclk SCLK pin of the SPI interface
   * @param cs CS pin connected to the chip
   * @param isr ISR pin connected to the chip
   */
  MuNFC(char appHash[16], uint32_t version,
      PinName mosi, PinName miso, PinName sclk, PinName cs, PinName isr);

  ~MuNFC();

  /** Set Encode Callback.
   *  The encode callback will be called on each start of NFC transaction.
   *  to populate the data structure that will be transmitted to the reader
   *  @param fn pointer to the function to be called
   *  @param arg argument that will be passed to the callback
   */
  inline void encode(void (*fn)(TLVList*, void*), void* arg)
  {
    m_encodeCb.attach(fn, arg);
  }

  /** Set Encode Callback.
   *  The encode callback will be called on each start of NFC transaction.
   *  to populate the data structure that will be transmitted to the reader
   *  @param inst pointer to the object on which to call the member
   *  @param member pointer to the object's member to be called
   */
  template <class T>
  inline void encode(T* inst, void (T::*member)(TLVList*))
  {
    m_encodeCb.attach(inst, member);
  }

  /** Set Decode Callback.
   *  The decode callback will be called on each successful termination of NFC transaction.
   *  populated with the data structure that was transmitted by the reader
   *  @param fn pointer to the function to be called
   *  @param arg argument that will be passed to the callback
   */
  inline void decode(void (*fn)(TLVList*, void*), void* arg)
  {
    m_decodeCb.attach(fn, arg);
  }

  /** Set Decode Callback.
   *  The decode callback will be called on each successful termination of NFC transaction.
   *  populated with the data structure that was transmitted by the reader
   *  @param inst pointer to the object on which to call the member
   *  @param member pointer to the object's member to be called
   */
  template <class T>
  inline void decode(T* inst, void (T::*member)(TLVList*))
  {
    m_decodeCb.attach(inst, member);
  }

  /** Set Event Callback.
   *  The event callback will be called on each of the following event:
   *  - Transaction started
   *  - Transaction successful
   *  - Transaction failed
   *  @param fn pointer to the function to be called
   *  @param arg argument that will be passed to the callback
   */
  inline void event(void (*fn)(NFCEvent, void*), void* arg)
  {
    m_eventCb.attach(fn, arg);
  }

  /** Set Event Callback.
   *  The event callback will be called on each of the following event:
   *  - Transaction started
   *  - Transaction successful
   *  - Transaction failed
   *  @param fn pointer to the function to be called
   *  @param arg argument that will be passed to the callback
   */
  template <class T>
  inline void event(T* inst, void (T::*member)(NFCEvent))
  {
    m_eventCb.attach(inst, member);
  }

  /** Initialize stack.
   *  @return true if stack was initialized correctly, false otherwise
   */
  bool init();

//#if MUNFC_RTOS -- flags must be disabled for proper doxygen support
  /** Start NFC thread (threaded mode)
   *
   */
  void run();
//#endif

#if MUNFC_RTOS
protected:
  /** NFC Thread
   *
   */
  void process();
#endif

#if MUNFC_RTOS
  protected:
#else
  public:
#endif
  /** Poll for NFC reader (polling mode).
   *  @param timeoutMs (maximum polling time)
   */
  void poll(int timeoutMs);

private:
#if MUNFC_RTOS
  static void staticCallback(void const* p);
#endif

  NdefCallback m_encodeCb;
  NdefCallback m_decodeCb;
  EventCallback m_eventCb;

  //DigitalIn m_irq_pin_int;
  InterruptIn m_irq_pin_isr;
  DigitalOut m_cs_pin;
  SPI m_spi;

#if MUNFC_RTOS
  Thread* m_pThread;
#endif

};


#endif /* MUNFC_H_ */
