#include "mbed.h"
#include "main.h"
// SX1272 Lib
#include "sx1272-hal.h"
#include "debug.h"
// DHT11 Lib (air temperature and humidity)
#include "DHT11.h"
// SHT10 (soil temperature and humidity)
#include "SHTx/sht15.hpp"

/* Set this flag to '1' to display debug messages on the console */
#define DEBUG_MESSAGE   1

/* DELAY between two transmission (in seconds) */
#define TRANSMISSION_DELAY                          1800

#define RF_FREQUENCY                                868000000 // Hz
#define TX_OUTPUT_POWER                             14        // 14 dBm

#define LORA_BANDWIDTH                              2         // [0: 125 kHz,
//  1: 250 kHz,
//  2: 500 kHz,
//  3: Reserved]

#define LORA_SPREADING_FACTOR                       7         // [SF7..SF12]
#define LORA_CODINGRATE                             1         // [1: 4/5,
//  2: 4/6,
//  3: 4/7,
//  4: 4/8]

#define LORA_PREAMBLE_LENGTH                        8
#define LORA_SYMBOL_TIMEOUT                         5         // Symbols
#define LORA_FIX_LENGTH_PAYLOAD_ON                  false
#define LORA_FHSS_ENABLED                           false
#define LORA_NB_SYMB_HOP                            4
#define LORA_IQ_INVERSION_ON                        false
#define LORA_CRC_ENABLED                            true

#define BUFFER_SIZE                                 6         // Define the payload size here

DigitalOut led(LED1);

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*
 *  Global variables declarations
 */
SX1272MB2xAS Radio( NULL );

uint8_t msg[BUFFER_SIZE];

uint16_t BufferSize = BUFFER_SIZE;
uint8_t Buffer[BUFFER_SIZE];

// Air temperature and humidity sensor
DHT11 airSensor(D6);
int DHT11_state;

// Soil temperature and humidity sensor
SHTx::SHT15 soilSensor(D9, D8);

int main()
{
    debug( "\n\n\r     iGreenhouse Application - Transmitter \n\n\r" );

    // Initialize Radio driver
    RadioEvents.TxDone = OnTxDone;
    RadioEvents.TxTimeout = OnTxTimeout;
    Radio.Init( &RadioEvents );

    // verify the connection with the board
    while( Radio.Read( REG_VERSION ) == 0x00  ) {
        debug( "Radio could not be detected!\n\r", NULL );
        wait( 1 );
    }

    debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1272MB2XAS ) ) , "\n\r > Board Type: SX1272MB2xAS < \n\r" );

    Radio.SetChannel( RF_FREQUENCY );


    debug_if( LORA_FHSS_ENABLED, "\n\n\r             > LORA FHSS Mode < \n\n\r");
    debug_if( !LORA_FHSS_ENABLED, "\n\n\r             > LORA Mode < \n\n\r");

    Radio.SetTxConfig( MODEM_LORA, TX_OUTPUT_POWER, 0, LORA_BANDWIDTH,
                       LORA_SPREADING_FACTOR, LORA_CODINGRATE,
                       LORA_PREAMBLE_LENGTH, LORA_FIX_LENGTH_PAYLOAD_ON,
                       LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                       LORA_IQ_INVERSION_ON, 2000000 );
                       
    // Soil sensor configuration
    soilSensor.setOTPReload(false);
    soilSensor.setResolution(true);
    soilSensor.setScale(false);

    debug_if( DEBUG_MESSAGE, "Starting sending loop\r\n" );

    led = 0;

    while(1) {
        // Retrieving sensors data
        DHT11_state = airSensor.readData();
        soilSensor.update();
        
        if (DHT11_state == DHT11::OK) {
            msg[0] = airSensor.readTemperature(); // Temperature - Air
            msg[1] = airSensor.readHumidity(); // Humidity - Air
        } else {
            msg[0] = 0x00; // Temperature - Air
            msg[1] = 0x00; // Humidity - Air
        }
        
        msg[2] = to_u8(soilSensor.getTemperature(), true); // Temperature - Soil
        msg[3] = to_u8(soilSensor.getHumidity(), false); // Humidity - Soil
        // Measurements types - Should be 0111 0010 -> 0x72 
        msg[4] = 0x72;
        // Greenhouse num 1 and sensors in the middle - Should be 0001 0001
        msg[5] = 0x11; // id serre
        
        // Sending a new packet
        debug("\r\n========\r\nSending a new Packet\r\n========\r\n");
        for(int i = 0; i < BufferSize; i++) {
            debug("%x", msg[i]);    
        }
        debug_if( DEBUG_MESSAGE, "\n" );
        memcpy( Buffer, msg, BufferSize );
        wait_ms(10);
        Radio.Send(Buffer, BufferSize);
        
        // Switch the led state
        led = 1-led;
        
        // wait X seconds before resend data
        //wait(TRANSMISSION_DELAY);
        wait(5);
    }
}

// temperature: -30 < x < 70
// humidity: 0 < x < 100
uint8_t to_u8(float x, bool isTemp)
{
  float a = 0;
  float min = 0.0;
  float max = 100.0;
  if( isTemp) {
    min = -30.0;
    max = 70.0;  
    a = 30.0;  
  }
  // On passe le float entre 0 et 1.0
  if(x > min && x < max) {
    x = (x + a) / 100.0;
  } else if(x <= min) {
    x = 0.0;
  } else {
    x = 1.0;
  }
  return rint(x * 255);
}

void OnTxDone( void )
{
    Radio.Sleep( );
    debug_if( DEBUG_MESSAGE, "> OnTxDone\n\r" );
}

void OnTxTimeout( void )
{
    Radio.Sleep( );
    debug_if( DEBUG_MESSAGE, "> OnTxTimeout\n\r" );
}

