#include "mbed.h"
#include "main.h"
#include "sx1272-hal.h"
#include "debug.h"

/* Set this flag to '1' to display debug messages on the console */
#define DEBUG_MESSAGE   1


#define RF_FREQUENCY                                868000000 // Hz

#define LORA_BANDWIDTH                              2         // [0: 125 kHz,
//  1: 250 kHz,
//  2: 500 kHz,
//  3: Reserved]

#define LORA_SPREADING_FACTOR                       7         // [SF7..SF12]
#define LORA_CODINGRATE                             1         // [1: 4/5,
//  2: 4/6,
//  3: 4/7,
//  4: 4/8]

#define LORA_PREAMBLE_LENGTH                        8
#define LORA_SYMBOL_TIMEOUT                         5         // Symbols
#define LORA_FIX_LENGTH_PAYLOAD_ON                  false
#define LORA_FHSS_ENABLED                           false
#define LORA_NB_SYMB_HOP                            4
#define LORA_IQ_INVERSION_ON                        false
#define LORA_CRC_ENABLED                            true

#define RX_TIMEOUT_VALUE                            3500000   // in us
#define BUFFER_SIZE                                 48        // Define the payload size here

DigitalOut led(LED1);

/*
 *  Global variables declarations
 */
typedef enum {
    LOWPOWER = 0,

    RX,
    RX_TIMEOUT,
    RX_ERROR
} AppStates_t;

volatile AppStates_t State = LOWPOWER;

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*
 *  Global variables declarations
 */
SX1272MB2xAS Radio( NULL );

uint16_t BufferSize = 0;
uint8_t Buffer[BUFFER_SIZE];

int16_t RssiValue = 0.0;
int8_t SnrValue = 0.0;

int main()
{
    debug( "\n\n\r     iGreenhouse Application - Receiver \n\n\r" );

    // Initialize Radio driver
    RadioEvents.RxDone = OnRxDone;
    RadioEvents.RxError = OnRxError;
    RadioEvents.RxTimeout = OnRxTimeout;
    Radio.Init( &RadioEvents );

    // verify the connection with the board
    while( Radio.Read( REG_VERSION ) == 0x00  ) {
        debug( "Radio could not be detected!\n\r", NULL );
        wait( 1 );
    }

    debug_if( ( DEBUG_MESSAGE & ( Radio.DetectBoardType( ) == SX1272MB2XAS ) ) , "\n\r > Board Type: SX1272MB2xAS < \n\r" );

    Radio.SetChannel( RF_FREQUENCY );


    debug_if( LORA_FHSS_ENABLED, "\n\n\r             > LORA FHSS Mode < \n\n\r");
    debug_if( !LORA_FHSS_ENABLED, "\n\n\r             > LORA Mode < \n\n\r");

    Radio.SetRxConfig( MODEM_LORA, LORA_BANDWIDTH, LORA_SPREADING_FACTOR,
                       LORA_CODINGRATE, 0, LORA_PREAMBLE_LENGTH,
                       LORA_SYMBOL_TIMEOUT, LORA_FIX_LENGTH_PAYLOAD_ON, 0,
                       LORA_CRC_ENABLED, LORA_FHSS_ENABLED, LORA_NB_SYMB_HOP,
                       LORA_IQ_INVERSION_ON, true );


    debug_if( DEBUG_MESSAGE, "Starting listening loop\r\n\r\n" );

    led = 0;
    Radio.Rx( RX_TIMEOUT_VALUE );

    while( 1 ) {
        switch( State ) {
            case RX:
                if( BufferSize > 0 ) {
                    debug_if( DEBUG_MESSAGE, "\r\n========\r\nNew Packet\r\n========\r\n" );
                    for(int i = 0; i < BufferSize; i++) {
                        debug_if( DEBUG_MESSAGE, "%x", Buffer[i]);    
                    }
                    debug_if( DEBUG_MESSAGE, "\r\n");
                    retrieve_data( Buffer );
                }
                BufferSize = 0;
                Radio.Rx( RX_TIMEOUT_VALUE );
                State = LOWPOWER;
                break;
            case RX_TIMEOUT:
                Radio.Rx( RX_TIMEOUT_VALUE );
                State = LOWPOWER;
                break;
            case RX_ERROR:
                // We have received a Packet with a CRC error
                Radio.Rx( RX_TIMEOUT_VALUE );
                State = LOWPOWER;
                break;
            case LOWPOWER:
                break;
            default:
                State = LOWPOWER;
                break;
        }
    }
}

void OnRxDone( uint8_t *payload, uint16_t size, int16_t rssi, int8_t snr)
{
    Radio.Sleep( );
    BufferSize = size;
    memcpy( Buffer, payload, BufferSize );
    RssiValue = rssi;
    SnrValue = snr;
    State = RX;
    debug_if( DEBUG_MESSAGE, "> OnRxDone %d \n\r", RssiValue );
}

void OnRxTimeout( void )
{
    Radio.Sleep( );
    BufferSize = 0;
    State = RX_TIMEOUT;
    debug_if( DEBUG_MESSAGE, "> OnRxTimeout\n\r" );
}

void OnRxError( void )
{
    Radio.Sleep( );
    State = RX_ERROR;
    debug_if( DEBUG_MESSAGE, "> OnRxError\n\r" );
}

float u8_to_float(uint8_t x, bool isTemp) 
{
    float a = 0.0;
    if ( isTemp ) {
        a = 30.0;
    }
    return (x/255.0)*100.0 - a;
}

void retrieve_data(uint8_t * payload)
{
    uint8_t measurements_type[4];
    uint8_t greenhouse_info = payload[5];
    printf("Greenhouse number: %d \r\n", greenhouse_info & 0x0F);
    printf("Sensors position: %d \r\n", (greenhouse_info & 0x30) >> 4);
    
    measurements_type[0] = payload[4] & 0x03;
    measurements_type[1] = (payload[4] >> 2) & 0x03;
    measurements_type[2] = (payload[4] >> 4) & 0x03;
    measurements_type[3] = (payload[4] >> 6) & 0x03;
    
    convert(payload[0], measurements_type[3]);
    convert(payload[1], measurements_type[2]);
    convert(payload[2], measurements_type[1]);
    convert(payload[3], measurements_type[0]);
}

void convert(uint8_t m, uint8_t t)
{
    if( t & 0x01 == 1 ) {
        // Air
        if( (t >> 1) & 0x01 == 1 ) {
            // Humidity
            printf("HumiA=%3.2f \r\n", (float) m);
        } else {
            // Temperature
            printf("TempA=%3.2f \r\n", (float) m);
        }
    } else {
        // Soil    
        if( (t >> 1) & 0x01 == 1 ) {
            // Humidity
            printf("HumiS=%3.2f \r\n", u8_to_float(m, false));
        } else {
            // Temperature
            printf("TempS=%3.2f \r\n", u8_to_float(m, true));
        }
    }
}