#ifndef plotly_streaming_ethernet_h
#define plotly_streaming_ethernet_h

#include <EthernetInterface.h>
#include <TCPSocketConnection.h>

// size of the large buffer used for constructing messages.
#define k_bufferSize 512


/** Create a plot on plot.ly
*
* Creates a streaming X/Y scatter plot with line on the plot.ly site.
* Multiple lines can be drawn however each line currently requires a seperate network socket so you could easily run out of resources if you're not careful.
* I've only tested 1 and 2 line plots.
*
* In theory each line could be plotted by a different mbed but this would require some changes to the way charts are initalised.
*
* Update periods can be between 50ms and 60s. Anything faster will be filtered, anything slower will result in the connection getting closed.
* 
* Requires an mbed with network support.
* 
* Based on the Ardunio code supplied by plot.ly and provided as is, it works for me but your mileage may vary.
* Sorry, I don't have time to offer much support on this.
* 
* You will need to create a plot.ly account and then go to https://plot.ly/settings to get your API key and streaming tokens.
* 
* See Plotly_HelloWorld for a sample implimentation.
* 
*/

class plotly
{
    public:
    /**
    @param username Your plot.ly username
    @param api_key Your plot.ly API key
    @param stream_tokens An array of plot.ly streaming token for your plot.ly account
    @param filename The name of the file to save the chart as
    @param nTraces The number of traces (MUST match the size of stream_tokens, can be omitted for a single line)
    */
        plotly(const char *username, const char *api_key, const char *stream_tokens[], const char *filename, int nTraces = 1);

        ~plotly();
        
        /** Initalises the chart
        
        This fucntion creates a blank chart on the plot.ly system and configures it to recieve streamed data using the specified token
        
        Time taken for this function can vary depending on network delays.
        
        If you wish to change any of the options like max points or world readability then make sure you change them BEFORE calling this function.       
        */
        bool init();

        /**
        Adds a point to the chart. The chart MUST be initalised before calling this.
        Note, if the streaming network port is closed then this will attempt to open the port and re-establish the stream connection, this could block for a while.
        @param x The X value.
        @param y The y value.
        @param stream Which trace to add the point to counting from 0, can be omitted for charts with a single line. 
        */
        void plot(unsigned long x, int y, int stream = 0);

        /**
        Adds a point to the chart. The chart MUST be initalised before calling this.
        Note, if the streaming network port is closed then this will attempt to open the port and re-establish the stream connection, this could block for a while.
        @param stream Which trace to add the point to counting from 0, can be omitted for charts with a single line. 
        @param x The X value.
        @param y The y value.
        */
        void plot(unsigned long x, float y, int stream = 0);

        /**
        Adds a point to the chart. The chart MUST be initalised before calling this.
        Note, if the streaming network port is closed then this will attempt to open the port and re-establish the stream connection, this could block for a while.
        
        @param x The X value.
        @param y The y value.
        @param stream Which trace to add the point to counting from 0, can be omitted for charts with a single line. 
        */
        void plot(float x, float y, int stream = 0);


        /**
        Opens all the streaming connections.
        
        Normally you'd do this after a sucessful call to init() and before starting plotting in order to avoid a connection delays when you first call plot()        
        */
        void openStreams();
        
        /** Close all the streaming connections
        
        Call to tidy up and free up system resources once there is no more data to send
        */
        void closeStreams();

        /**
        Opens a specific streaming connection. Normally you would use openStreams() however if you only want to update a single line on a multi-line chart then this can save time and memory.
        
        @param stream The line number (from 0) to open
        */
        bool openStream(int stream);
        
        /**
        Close a specific streaming connection.
        
        @param stream The line number (from 0) to open
        */
        void closeStream(int stream);

        /**
         output message level
         Messages are sent to stderr (which defaults to the mBed USB programing/debug port).
          0 = Debugging, 1 = Informational, 2 = Status, 3 = Errors (default), 4 = Quiet
          */
        int log_level;

        /**
        set true to not actually connect to the network..
        */
        bool dry_run;

        /**
        Maximum points to display on the streaming chart, once you go over this old points will no longer be displayed.
        Defaults to 30
        */
        int maxpoints;
        
        /**
        Sets whether the chart will be public or not.
        Defaults to true
        */
        bool world_readable;
        
        /**
        Converts timestamps to the local time zone
        */
        bool convertTimestamp;
        /**
        Timezone string to use
        */
        char *timezone;
        /**
        Sets what to do if the file already exists, valid options are:
        "new","overwrite" (default),"append","extend"
        */
        char *fileopt;

    private:

        void reconnectStream(int stream);

        bool print_(int d,int stream = 0);
        bool print_(unsigned long d,int stream = 0);
        bool print_(float d,int stream = 0);
        bool print_(char *d,int stream = 0) {return print_((const char *)d, stream);};
        bool print_(const char *d,int stream = 0);
        bool printHex_(uint16_t d,int stream = 0);
        bool printNetTerminator_(int stream = 0);

        bool sendFormatedText(char* data, int size,int stream = 0);

        char buffer[k_bufferSize];

        TCPSocketConnection **sockets;
        
        const char *username_;
        const char *api_key_;
        const char** stream_tokens_;
        const char *filename_;
        int nTraces_;
        
        bool initalised;

};
#endif
