#include "plotly.h"
#include "mbed.h"

#define plotlyURL "plot.ly"
#define dataURL "arduino.plot.ly"

plotly::plotly(const char *username, const char *api_key, const char* stream_tokens[], const char *filename, int nTraces)
{
    log_level = 3;  // 0 = Debugging, 1 = Informational, 2 = Status, 3 = Errors, 4 = Quiet (// Serial Off)
    dry_run = false;
    username_ = username;
    api_key_ = api_key;
    stream_tokens_ = stream_tokens;
    filename_ = filename;
    maxpoints = 30;
    world_readable = true;
    convertTimestamp = false;
    timezone = "America/Montreal";
    fileopt = "overwrite";

    nTraces_ = nTraces;

    sockets = (TCPSocketConnection **)malloc(sizeof(TCPSocketConnection *) * nTraces_);
    for (int i = 0; i< nTraces_; i++) {
        *(sockets+i) = NULL;
    }
    initalised = false;

}


plotly::~plotly()
{
    closeStreams();
    if (sockets)
        free(sockets);
}


bool plotly::init()
{
    //
    //  Create plot with a REST post to plotly
    //  See the clientresp section of https://plot.ly/rest/ for details
    //
    if(dry_run && log_level < 3) {
        fprintf(stderr,"... This is a dry run, we are not connecting to plotly's servers...\n");
    } else if(log_level < 3) {
        fprintf(stderr,"... Attempting to connect to plotly's REST servers\n");
    }

    if (!dry_run) {
        int pause = 1;
        *sockets = new TCPSocketConnection();
        if (!*sockets)
            return false;
        while ((*sockets)->connect(plotlyURL, 80) < 0) {
            if (pause > 30) {
                fprintf(stderr,"Failed to connect. :-(\n");
                delete (*sockets);
                return false;
            }
            fprintf(stderr,"... Couldn\'t connect to plotly's REST servers... trying again!\n");
            wait(pause);
            pause *= 2;
        }
    }

    if(log_level < 3) fprintf(stderr,"... Connected to plotly's REST servers\n");

    if(log_level < 3) fprintf(stderr,"... Sending HTTP Post to plotly\n");

    print_("POST /clientresp HTTP/1.1\r\n");
    print_("Host: 107.21.214.199\r\n");
    print_("User-Agent: Arduino/0.5.1\r\n");
    print_("plotly-streamtoken: ");
    for (int i = 0; i<nTraces_; i++) {
        print_(stream_tokens_[i]);
        if ( i < (nTraces_-1) )
            print_(",");
    }
    printNetTerminator_();

    print_("Content-Length: ");

    unsigned long lineLen = snprintf(buffer,k_bufferSize,"version=2.3&origin=plot&platform=arduino&un=%s&key=%s&args=[",username_,api_key_);
    for(int i=0; i<nTraces_; i++) {
        lineLen += snprintf((buffer+lineLen),k_bufferSize-lineLen,"{\"y\": [], \"x\": [], \"type\": \"scatter\", \"stream\": {\"token\": \"%s\", \"maxpoints\": %d}}%s",stream_tokens_[i],maxpoints,((nTraces_ > 1) && (i != nTraces_-1))?", ":"");
    }
    lineLen += snprintf((buffer+lineLen),k_bufferSize-lineLen,"]&kwargs={\"fileopt\": \"%s\", \"filename\": \"%s\", \"world_readable\": %s}",fileopt,filename_,world_readable?"true":"false");

    print_(lineLen);
    printNetTerminator_();
    printNetTerminator_();

    sendFormatedText(buffer,lineLen);

    printNetTerminator_();

    //
    // Wait for a response
    // Parse the response for the "All Streams Go!" and proceed to streaming
    // if we find it
    //

    char allStreamsGo[] = "All Streams Go!";
    int asgCnt = 0; // asg stands for All Streams Go
    char url[] = "\"url\": \"http://107.21.214.199/~";
    char fid[4];
    int fidCnt = 0;
    int urlCnt = 0;
    int usernameCnt = 0;
    bool proceed = false;
    bool fidMatched = false;
    char c;

    if(log_level < 2) {
        fprintf(stderr,"... Sent message, waiting for plotly's response...\n");
    }

    if(!dry_run) {
        while(!proceed) {
            int32_t dataIn = (*sockets)->receive(buffer,k_bufferSize -1);
            if (dataIn < 0) {
                if(log_level < 3) fprintf(stderr,"error reading network socket\n");
                break;
            }
            if(dataIn > 0) {
                buffer[dataIn]=0;

                if(log_level < 2) fprintf(stderr,buffer);

                for (int i = 0; i<dataIn; i++) {
                    c = buffer[i];
                    //
                    // Attempt to read the "All streams go" msg if it exists
                    // by comparing characters as they roll in
                    //

                    if(asgCnt == strlen(allStreamsGo) && !proceed) {
                        proceed = true;
                    } else if(allStreamsGo[asgCnt]==c) {
                        asgCnt += 1;
                    } else if(asgCnt > 0) {
                        // reset counter
                        asgCnt = 0;
                    }

                    //
                    // Extract the last bit of the URL from the response
                    // The url is in the form http://107.21.214.199/~USERNAME/FID
                    // We'll character-count up through char url[] and through username_, then start
                    // filling in characters into fid
                    //

                    if(log_level < 3) {
                        if(url[urlCnt]==c && urlCnt < strlen(url)) {
                            urlCnt += 1;
                        } else if(urlCnt > 0 && urlCnt < strlen(url)) {
                            // Reset counter
                            urlCnt = 0;
                        }
                        if(urlCnt == strlen(url) && fidCnt < 4 && !fidMatched) {
                            // We've counted through the url, start counting through the username
                            if(usernameCnt < strlen(username_)+2) {
                                usernameCnt += 1;
                            } else {
                                // the url ends with "
                                if(c != '"') {
                                    fid[fidCnt] = c;
                                    fidCnt += 1;
                                } else if(fidCnt>0) {
                                    fidMatched = true;
                                }

                            }
                        }
                    }
                }
            }
            wait(0.1);
        }
    }

    if(!dry_run && !proceed && log_level < 4) {
        fprintf(stderr,"... Error initializing stream, aborting.\n");
    }

    if(!dry_run && proceed && log_level < 3) {
        fprintf(stderr,"... A-ok from plotly, All Streams Go!\n");
        if(fidMatched) {
            fprintf(stderr,"... View your streaming plot here: https://plot.ly/~");
            fprintf(stderr,username_);
            fprintf(stderr,"/");
            for(int i=0; i<fidCnt; i++) {
                fprintf(stderr,"%c",fid[i]);
            }
            fprintf(stderr,"\n");
        }
    }

    if (proceed || dry_run) {
        initalised = true;
    }
    if (*sockets) {
        delete *sockets;
        *sockets=NULL;
    }
    return initalised;
}

void plotly::openStreams()
{
    for (int i = 0; i< nTraces_; i++) {
        openStream(i);
    }
}

void plotly::closeStreams()
{
    for (int i = 0; i< nTraces_; i++) {
        closeStream(i);
    }
}


bool plotly::openStream(int stream)
{

    if (stream >= nTraces_)
        return false;

    if (!initalised)
        return false;

    //
    // Start request to stream servers
    //

    if (*(sockets+stream) != NULL) {
        delete *(sockets+stream);
        *(sockets+stream) = NULL;
    }

    if(log_level < 3) fprintf(stderr,"... Connecting to plotly's streaming servers...\n");

    if (!dry_run) {
        int pause = 1;
        *(sockets + stream) = new TCPSocketConnection();
        while ((*(sockets + stream))->connect(dataURL, 80) < 0) {
            if (pause > 30) {
                fprintf(stderr,"Failed to connect. :-(\n");
                delete *(sockets + stream);
                return false;
            }
            fprintf(stderr,"... Couldn\'t connect to plotly's REST servers... trying again!\n");
            wait(pause);
            pause *= 2;
        }
    }


    if(log_level < 3) fprintf(stderr,"... Connected to plotly's streaming servers\n... Initializing stream %d\n",stream);

    print_("POST / HTTP/1.1\r\n",stream);
    print_("Host: arduino.plot.ly\r\n",stream);
    print_("User-Agent: Python\r\n",stream);
    print_("Transfer-Encoding: chunked\r\n",stream);
    print_("Connection: close\r\n",stream);
    print_("plotly-streamtoken: ",stream);
    print_(stream_tokens_[stream],stream);
    printNetTerminator_(stream);
    if(convertTimestamp) {
        print_("plotly-convertTimestamp: \"",stream);
        print_(timezone,stream);
        print_("\"",stream);
        printNetTerminator_(stream);
    }
    printNetTerminator_(stream);

    if(log_level < 3) fprintf(stderr,"... Done initializing, ready to stream!\n");
    return true;
}

void plotly::closeStream( int stream)
{
    if (stream >= nTraces_)
        return;

    if (*(sockets+stream) != NULL) {
        if ((*(sockets+stream))->is_connected()) {
            print_("0\r\n\r\n",stream);
            (*(sockets+stream))->close();
        }
        delete *(sockets+stream);
        *(sockets+stream)=NULL;
    }
}

void plotly::reconnectStream(int number)
{
    while(!dry_run && (!(*(sockets+number)) || !(*(sockets+number))->is_connected())) {
        if(log_level<4) fprintf(stderr,"... Disconnected from streaming servers\n");
        closeStream(number);
        openStream(number);
    }
}


void plotly::plot(unsigned long x, int y, int stream)
{
    if (!initalised)
        return;

    reconnectStream(stream);

    // need to prefix with the length so print it once to get the content lenght and then a second time with the prefix.
    int len = snprintf(buffer,k_bufferSize,"{\"x\": %lu, \"y\": %d}\n", x,y);
    len = snprintf(buffer,k_bufferSize,"%x\r\n{\"x\": %lu, \"y\": %d}\n\r\n",len, x,y);
    sendFormatedText(buffer,len,stream);
}

void plotly::plot(unsigned long x, float y, int stream)
{
    if (!initalised)
        return;

    reconnectStream(stream);

    // need to prefix with the length so print it once to get the content lenght and then a second time with the prefix.
    int len = snprintf(buffer,k_bufferSize,"{\"x\": %lu, \"y\": %.3f}\n", x,y);
    len = snprintf(buffer,k_bufferSize,"%x\r\n{\"x\": %lu, \"y\": %.3f}\n\r\n",len, x,y);
    sendFormatedText(buffer,len,stream);
}

void plotly::plot(float x, float y, int stream)
{
    if (!initalised)
        return;

    reconnectStream(stream);
    // need to prefix with the length so print it once to get the content lenght and then a second time with the prefix.
    int len = snprintf(buffer,k_bufferSize,"{\"x\": %.3f, \"y\": %.3f}\n", x,y);
    len = snprintf(buffer,k_bufferSize,"%x\r\n{\"x\": %.3f, \"y\": %.3f}\n\r\n",len, x,y);
    sendFormatedText(buffer,len,stream);
}


bool plotly::print_(int d, int stream)
{
    char smallBuffer[10];
    int32_t len = snprintf(smallBuffer,10,"%d",d);
    return sendFormatedText(smallBuffer,len,stream);
}

bool plotly::print_(unsigned long d, int stream)
{
    char smallBuffer[12];
    int32_t len = snprintf(smallBuffer,12,"%lu",d);
    return sendFormatedText(smallBuffer,len,stream);
}

bool plotly::print_(float d, int stream)
{
    char smallBuffer[12];
    int32_t len = snprintf(smallBuffer,12,"%f",d);
    return sendFormatedText(smallBuffer,len,stream);
}

bool plotly::printNetTerminator_(int stream)
{
    return sendFormatedText("\r\n",2,stream);
}



bool plotly::print_(const char *d, int stream)// strings could be long, use the big buffer
{
    int32_t len = snprintf(buffer,k_bufferSize,"%s",d);
    return sendFormatedText(buffer,len,stream);
}

bool plotly::printHex_(uint16_t d, int stream)
{
    int32_t len = snprintf(buffer,k_bufferSize,"%X",d);
    return sendFormatedText(buffer,len,stream);
}

bool plotly::sendFormatedText(char* data, int size, int stream)
{
    if(log_level < 2) {
        fprintf(stderr,"%s",data);
    }
    if(!dry_run) {
        if (!*(sockets+stream)) {
            fprintf(stderr,"\nTX failed, No network socket exists\n");
            return false;
        }
        if (!((*(sockets+stream))->is_connected())) {
            fprintf(stderr,"\nTX failed, Network socket not connected\n");
            return false;
        }

        int32_t sent = (*(sockets+stream))->send_all(data,size);
        if (sent == size)
            return true;
        else {
            fprintf(stderr,"\nTX failed to send _%s_ Sent %d of %d bytes\n",data,sent,size);
            return false;
        }
    } else
        return true;
}

