#ifndef __miniFont_h__
#define __miniFont_h__

#include "mbed.h"
 
/** A basic low resolution font
** This font is intened for very low resolution display systems e.g. LED matrix displays.
*
* All characters are 5 pixels high and 5 or less pixels wide.
*
* Data for a given character is returned as an array of 5 chars
* each char represents one row of the image, the first being the top row.
* The least significant 5 bits of each char represent the pixels on that line.
*
* e.g. For the letter X the returnd data is
* { 0x11, 0x0a, 0x04, 0x0a, 0x11 }
* As bits this translates to
*
* 10001
*
* 01010
*
* 00100
*
* 01010
*
* 10001
*
* example: The following code will output the bit pattern for the selected character.
*
* @code
*
* #include "miniFont.h"
* miniFont myFont;
*
* int main() {
*    
*    char characterData[5];
     char letter = 'x'
*    if (myFont.getChar(letter,&characterData)) {
*        for (int row = 0; row < myFont.getPixHeight(letter); row++) {
           for (int col = myFont.getPixWidth(letter) - 1; col >= 0; col--) { // start at the left no the right
             if (characterData[row] & (0x01<<col)) 
               printf("1");
             else
               printf("0");
             }
             printf("\n");
           }
*        }
*    }
* @endcode
*
* Characters are right justified, e.g. if getWidth() returns a value of 3 then only the 3 least significant bits of each row contain data.
*
* There are options to force fixed width characters (narrow characers will be shifted to the middle of a 5 pixel width) and to rotate characters 90 degrees
*
* Supported characters are A-Z 0-9 . ! ? <space>
*
* Any lower case letters are automatically capitalised.
*
* Not tested excessivly, if you find any problems please let me know.
*/
class miniFont {
public:

/** Initalise the font
*/
    miniFont();

/** Returns the width of the requested character
@param letter The character to measure
@return The width in pixels
*/    
    uint8_t getPixWidth(char letter);

/** Returns the height of the requested character
@param letter The character to measure
@return The height in pixels (currently always 5)
*/    
    uint8_t getPixHeight(char letter);

/** get the data for a given character
@param letter The chatacter to retrieve the data for
@param rowArray An array containing the pixel data
@return Flag indicating that rowArray is valid.
*/
    bool getChar(char letter, const char **rowArray);
    
/** get current fixed width mode
@return true if fixed width is enabled
*/
    bool isFixedWidth() {return fixedWidth;};
/** set fixed width mode
@param newValue True to enable fixed width, false to disable
*/
    void setFixedWidth(bool newValue) {fixedWidth = newValue;};
    
/** get current rotation mode
@return true if font is rotated
*/
    bool isRotated() {return rotate90;};

/** set rotation mode
@param newValue True to rotate text 90 degrees anti-clockwise (except in the USA where it rotates counterclockwise instead)
*/
    void setRotated(bool newValue) {rotate90 = newValue;};

private:

    uint8_t getMinPixWidth(char letter);
    bool getRawChar(char letter, const char **rowArray);
    bool getVerticalChar(char letter, const char **rowArray);

    static const uint8_t maxWidth = 5;
    static const uint8_t maxHeight = 5;

    bool rotate90;
    bool fixedWidth;
    char letterBuffer[5];
    char rotateBuffer[5];


    static const char letters[26][5];

    static const char numbers[10][5];

    static const char dot[5];
    static const char space[5];
    static const char exclam[5];
    static const char quest[5];

};


#endif
