#ifndef __DW1000SETUP_H__
#define __DW1000SETUP_H__

#include "mbed.h"
/** Class for holding DW1000 config options
*
* Defaults are channel 5, 16MHz PRF, 850kb/s, standard SFD, 128 symbol preamble, preamble code 3.
* All pins are GPIO other than sync and IRQ.
* Smart power is off.
*
*/
class DW1000Setup
{
public:

    /** Supported UWB modes
    *
    *  enum for preset mode combinations
    */
    typedef enum {fastLocationC5, ///< maximum data speed, short range. Channel 5
                  fastLocationC4, ///< maximum data speed, short range. Channel 4
                  tunedDefault,   ///< The power up default config with the recomended changes
                  user110k,       ///< 110kb/s settings used by Matthias Grob & Manuel Stalder
                  rangeRateCompromise ///< 850kb/s settings aimed to give a short packet but with far greater range than the fastLocation options
                 } UWBMode;

    /** Constructor
    * @param modeToUse The operating mode to set the initial settings to.
    *
    */
    DW1000Setup(UWBMode modeToUse = tunedDefault);


/// copy constructor
    DW1000Setup(DW1000Setup *orij);

/// Update this config to match the supplied one
    void applyConfig(DW1000Setup *orij);


/// enum for PRF options
    enum prf_e {prf16MHz, ///< PRF rate of 16MHz. Lower power
                prf64MHz ///< PRF rate of 64MHz. Higher power but more accurate timing.
               };

/// enum for data rate options
    enum dataRate_e {kbps110, ///< Data rate of 110kb/s (non-standard)
                     kbps850,///< Data rate of 850kb/s
                     kbps6800///< Data rate of 6.8Mb/s
                    };

    /** enum for SFD options
    *
    * To conform to the standard the standard setting should be used.
    * The decawave setting claimes to give better performance. Doesn't work for me at 850kb/s but does at 6.8Mb/s
    */
    enum sfd_e {standard, ///< IEEE standard SFD
                decaWave, ///< Decawave defined SFD
                user ///< user defined SFD
               };

/// enum for preamble length options
    enum preamble_e { pre64,///< Preamble is 64 symbols
                      pre128,///< Preamble is 128 symbols (non-standard)
                      pre256,///< Preamble is 256 symbols (non-standard)
                      pre512,///< Preamble is 512 symbols (non-standard)
                      pre1024,///< Preamble is 1024 symbols
                      pre1536,///< Preamble is 1536 symbols (non-standard)
                      pre2048, ///< Preamble is 2048 symbols (non-standard)
                      pre4096///< Preamble is 4096 symbols
                    };

    /** Set the Transmit power for smartpower operation
    * @param powerdBm The gain setting in dBm (0-33.5)
    * @param power500us The gain setting packets under 500us in dBm (0-33.5)
    * @param power250us The gain setting packets under 250us in dBm (0-33.5)
    * @param power125us The gain setting packets under 125us in dBm (0-33.5)
    * @return true if a valid option
    */
    bool setSmartTxPower(float powerdBm,float power500us=0,float power250us=0,float power125us=0) ;

    /** Set the Transmit power for non-smartpower operation
    * @param powerdBm The gain setting in dBm for the main packet(0-33.5)
    * @param powerPRF The gain setting in dBm for the preamble (0-33.5)
    * @return true if a valid option
    */
    bool setTxPower(float powerdBm,float powerPRF=0);


    /** Set the PRF
    * @return true if a valid option
    */
    bool setPRF(enum prf_e newSetting) {
        prf = newSetting;
        return true;
    };

    /** Set the Channel
    * @return true if a valid option
    *
    * channels are 1-5 & 7
    */
    bool setChannel(unsigned char newChannel) {
        if ((newChannel > 0) && ((newChannel <= 5) || (newChannel == 7))) {
            channel = newChannel;
            return true;
        }
        return false;
    };
    /** Set the SFD
    * @return true if a valid option
    */
    bool setSfd(enum sfd_e newSetting) {
        sfd = newSetting;
        return true;
    };
    /** Set the Preamble length
    * @return true if a valid option
    *
    * For 6.8Mb/s it should be 64 to 256 symbols.
    * For 850kb/s it should be 256 to 2048 symbols
    * For 110kb/s it should be >= 2048 symbols.
    */
    bool setPreambleLength(enum preamble_e newSetting) {
        preamble = newSetting;
        return true;
    };
    /** Set the Data rate
    * @return true if a valid option
    */
    bool setDataRate(enum dataRate_e newSetting)  {
        dataRate = newSetting;
        return true;
    };
    /** Set the Preamble code
    * @return true if a valid option
    *
    * @note Not all codes are valid for all channels, see the table below

<table>
<tr><th>channel</th><th>16MHzPrf</th><th>64MHzPrf</th></tr>
<tr><td>1</td><td>1, 2</td><td>9, 10, 11, 12</td></tr>
<tr><td>2</td><td>3, 4</td><td>9, 10, 11, 12</td></tr>
<tr><td>3</td><td>5, 6</td><td>9, 10, 11, 12</td></tr>
<tr><td>4</td><td>7, 8</td><td>17, 18, 19, 20</td></tr>
<tr><td>5</td><td>3, 4</td><td>9, 10, 11, 12</td></tr>
<tr><td>7</td><td>7, 8</td><td>17, 18, 19, 20</td></tr>
</table>

    */
    bool setPreambleCode(unsigned char newCode) {
        if ((newCode > 0) && (newCode <= 24)) {
            preambleCode = newCode;
            return true;
        }
        return false;
    };
    /** Set the smartpower state
    * @return true if a valid option
    *
    * only takes effect at 6.8Mb/s
    */
    bool setSmartPower(bool enable) {
        enableSmartPower = enable;
        return true;
    };


    /** Get the current transmit gains
    * @return An array containing the power levels
    *
    * An array of 4 floats is returned.
    * For smart power these are the 4 power levels (normal, <500us, <250us, <125us).
    * For non-smart power the first and last values are ignored, the second is the power for the phy header, the 3rd is the power for the reaminder of the frame.
    */
    const float *getTxPowers() {
        return powers;
    }

    /** Get the current channel
    * @return the channel number
    */
    unsigned char getChannel() {
        return channel;
    };
    /** Get the current PRF
    * @return the PRF
    */
    enum prf_e getPRF() {
        return prf;
    };
    /** Get the current data rate
    * @return the data rate
    */
    enum dataRate_e getDataRate() {
        return dataRate;
    };

    /** Get the current SFD mode
    * @return the SFD
    */
    enum sfd_e getSfd() {
        return sfd;
    };
    /** Get the current preamble length
    * @return the  preamble length
    */
    enum preamble_e getPreambleLength() {
        return preamble;
    };
    /** Get the current preamble code
    * @return the preamble code
    */
    unsigned char getPreambleCode() {
        return preambleCode;
    };
    /** Get the current smart power mode
    * @return true if smartpower is on
    */
    bool getSmartPower() {
        return  enableSmartPower;
    };

    /** Set which pins are GPIO
    *
    *  @param pins A bitmask of which pins are not to be used as GPIO
    *
    *  e.g. 0x01 sets GPIO0 as it's alternative function (RXOKLED) and all other pins to be GPIO.
    *
    *  The default is 0x0180 - GPIO 8 = irq, GPIO 7 = sync. All others are GPIO functionality
    */
    void setGPIO(uint16_t pins) {
        GPIO_Pins = pins;
    };

    /** Get which pins are GPIO
    *
    *  @return A bitmask of which pins are not to be used as GPIO
    *
    *  e.g. 0x01 Indicates that GPIO0 as it's alternative function (RXOKLED) and all other pins to be GPIO.
    *
    *  The default is 0x0180 - GPIO 8 = irq, GPIO 7 = sync. All others are GPIO functionality
    */
    uint16_t getGPIO() {
        return GPIO_Pins;
    };

    /** Check that the settings are self consistent
    *
    * @return true if the settings are valid
    *
    * Will check the following:
    *  - Preamble size is sensible for the data rate
    *  - Preamble code is valid for the channel
    *  - That smart power is only enabled at 6.8Mb/s
    */
    bool check();

    /** Get setup description
    *
    * @param buffer Data buffer to place description in
    * @param len Length of data buffer
    *
    * Places a text string describing the current setup into the suppled buffer.
    */
    void getSetupDescription(char *buffer, int len);

private:
    float powers[4];
    uint16_t GPIO_Pins;
    unsigned char channel; // 1-5 , 7
    enum prf_e prf;
    enum dataRate_e dataRate;
    enum sfd_e sfd;
    enum preamble_e preamble;
    unsigned char preambleCode; // 1-24. See section 10.5 of user manual for details.
    bool enableSmartPower;
};

#endif
