
#include "DW1000Setup.h"

DW1000Setup::DW1000Setup(UWBMode modeToUse)
{
    setGPIO(0x0180);
    switch (modeToUse) {
        case user110k:  // values from Matthias Grob & Manuel Stalder - ETH Zürich - library

            setChannel(5);
            setPRF(prf16MHz);
            setDataRate(kbps110);
            setSfd(standard);
            setPreambleLength(pre1024);
            setPreambleCode(3);
            setSmartPower(false);
            setTxPower(18);
            break;
        case tunedDefault:    // User Manual "2.5.5 Default Configurations that should be modified" p. 22
        default:
            setChannel(5);
            setPRF(prf16MHz);
            setDataRate(kbps6800);
            setSfd(standard);
            setPreambleLength(pre128);
            setPreambleCode(3);
            setSmartPower(false);
            setTxPower(18);
            break;
        case fastLocationC5:
            setChannel(5);
            setPRF(prf64MHz);
            setDataRate(kbps6800);
            setSfd(standard);
            setPreambleLength(pre64);
            setPreambleCode(10);
            setSmartPower(true);
            setTxPower(18);
            break;
        case fastLocationC4:
            setChannel(4);
            setPRF(prf64MHz);
            setDataRate(kbps6800);
            setSfd(standard);
            setPreambleLength(pre64);
            setPreambleCode(18);
            setSmartPower(true);
            setTxPower(18);
            break;
        case rangeRateCompromise:
            setChannel(4);
            setPRF(prf64MHz);
            setDataRate(kbps850);
            setSfd(standard);  // for some reason decawave doesn't work.
            setPreambleLength(pre256);
            setPreambleCode(18);
            setSmartPower(false);
            setTxPower(18);
            break;
    }
}


DW1000Setup::DW1000Setup(DW1000Setup *orij) {
        channel = orij->getChannel();
        prf = orij->getPRF();
        dataRate = orij->getDataRate();
        sfd = orij->getSfd();
        preamble = orij->getPreambleLength();
        preambleCode = orij->getPreambleCode();
        GPIO_Pins = orij->getGPIO(); // 8 = irq, 7 = sync. Others are GPIO
        enableSmartPower = orij->getSmartPower();
        memcpy(&powers,orij->getTxPowers(),4*sizeof(float));
    }


    void DW1000Setup::applyConfig(DW1000Setup *orij) {
        channel = orij->getChannel();
        prf = orij->getPRF();
        dataRate = orij->getDataRate();
        sfd = orij->getSfd();
        preamble = orij->getPreambleLength();
        preambleCode = orij->getPreambleCode();
        GPIO_Pins = orij->getGPIO(); // 8 = irq, 7 = sync. Others are GPIO
        enableSmartPower = orij->getSmartPower();
        memcpy(&powers,orij->getTxPowers(),4*sizeof(float));
    }


bool DW1000Setup::check()
{
    int maxCode = 24;
    int minCode = 1;

    if (prf == prf16MHz)
        switch (channel) {
            case 1:
                maxCode = 2;
                minCode = 1;
                break;
            case 2:
            case 5:
                maxCode = 4;
                minCode = 3;
                break;
            case 3:
                maxCode = 6;
                minCode = 5;
                break;
            case 4:
            case 7:
                maxCode = 8;
                minCode = 7;
                break;
            default:
                return false; // invalid channel
        }
    else
        switch (channel) {
            case 1:
            case 2:
            case 3:
            case 5:
                maxCode = 12;
                minCode = 9;
                break;
            case 4:
            case 7:
                maxCode = 20;
                minCode = 17;
                break;
            default:
                return false; // invalid channel
        }

    if ((preambleCode > maxCode) || (preambleCode < minCode))
        return false;

    switch (preamble) {
        default:
            return false;
        case pre64:
        case pre128:
            if (dataRate != kbps6800)
                return false;
            break;
        case pre256:
            if (dataRate == kbps110)
                return false;
            break;
        case pre512:
        case pre1024:
        case pre1536:
            if (dataRate != kbps850)
                return false;
            break;
        case pre2048:
            if (dataRate == kbps6800)
                return false;
            break;
        case pre4096:
            if (dataRate != kbps110)
                return false;
            break;
    }

    if (enableSmartPower) {
        if (dataRate != kbps6800)
            return false;
    }
    return true;
}

void DW1000Setup::getSetupDescription(char *buffer, int len)
{

    char dataRateString[10];
    if (dataRate == kbps6800)
        strcpy(dataRateString,"6.8 Mbps");
    else if (dataRate == kbps850)
        strcpy(dataRateString,"850 kbps");
    else
        strcpy(dataRateString,"110 kbps");

    char preambleString[8];
    switch (preamble) {
        default:
            strcpy(preambleString,"error");
            break;
        case pre64:
            strcpy(preambleString,"64");
            break;
        case pre128:
            strcpy(preambleString,"128");
            break;
        case pre256:
            strcpy(preambleString,"256");
            break;
        case pre512:
            strcpy(preambleString,"512");
            break;
        case pre1024:
            strcpy(preambleString,"1024");
            break;
        case pre1536:
            strcpy(preambleString,"1536");
            break;
        case pre2048:
            strcpy(preambleString,"2048");
            break;
        case pre4096:
            strcpy(preambleString,"4096");
            break;
    }

    snprintf(buffer,len,"Channel:\t%u\r\nPRF:\t%s\r\nData Rate:\t%s\r\nPreamble length:\t%s\r\nPreamble code:\t%u\r\nSmart power:\t%s\r\nSFD:\t%s\r\nTx Gain:\t%.1f,%.1f,%.1f,%.1f\r\n",
             channel,
             (prf == prf16MHz)?"16 MHz":"64 MHz",
             dataRateString,
             preambleString,
             preambleCode,
             enableSmartPower?"Enabled":"Disabled",
             (sfd == standard)?"Standard":"Non-standard",
             powers[0],powers[1],powers[2],powers[3]);
}

    bool DW1000Setup::setSmartTxPower(float powerdBm,float power500us,float power250us,float power125us) {
        if ((powerdBm < 0) || (powerdBm > 33.5))
            return false;

        powers[0] = powerdBm;

        if ((power500us < powerdBm) || (power500us > 33.5))
            powers[1] = powers[0];
        else
            powers[1] = power500us;

        if ((power250us < power500us) || (power250us > 33.5))
            powers[2] = powers[1];
        else
            powers[2] = power250us;

        if ((power125us < power250us) || (power125us > 33.5))
            powers[3] = powers[2];
        else
            powers[3] = power125us;

        return true;
    }

    bool DW1000Setup::setTxPower(float powerdBm,float powerPRF) {
        if ((powerdBm < 0) || (powerdBm > 33.5))
            return false;

        powers[0] = powerdBm;
        powers[1] = powerdBm;
        powers[2] = powerdBm;
        powers[3] = powerdBm;

        if ((powerPRF > 0) && (powerPRF < 33.5))
            powers[1] = powerPRF;

        return true;
    }
