#include "mbed.h"
#include "sgam_mdw.h"
#include "sgam_mdw_impl.h"

#include "CayenneLPP.h"

Serial pc(USBTX, USBRX);                    // SAIDA SERIAL PADRÃO
// #define D_LOG(args...)   pc.printf(args) 

#define ONE_SECOND      1

// LORA WAN CONFIGS 
#define TX_INTERVAL             10000
#define MBED_CONF_LORA_APP_PORT 15 //15

static uint8_t LORAWAN_DEV_EUI[] = { 0x00, 0x1C, 0x73, 0x4A, 0x55, 0x89, 0xAE, 0xC6 };
static uint8_t LORAWAN_APP_EUI[] = { 0x70, 0xB3, 0xD5, 0x7E, 0xD0, 0x01, 0xD0, 0x2F };
static uint8_t LORAWAN_APP_KEY[] = { 0x75, 0xC6, 0xF2, 0xFB, 0xEE, 0xA9, 0x82, 0x6C, 0xA0, 0xBD, 0xB9, 0x0F, 0xC9, 0xEC, 0xF7, 0x10 };

////////////////////////////////////////////////////////////////
static void lora_event_handler(lorawan_event_t event);
static int16_t count_message;

static lorawan_app_callbacks_t callbacks;

ControlImpl ctrl;
Communication<LoraData>* comm = NULL;

////////////////////////////////////////////////////////////////
int inicializa_lorawan(Control* control);

void read_send_gyroscope(void const *self_context);
void read_send_temperature(void const *self_context);
void read_send_GPS(void const *self_context);
void inicializa_thread_GPS(Control* control);

void finalize_sending(Communication<void*>* COMM, CayenneLPP* payload);
static void receive_message();

// FILAS PARA EVENTOS
EventQueue ev_queue; // Usado no callback do lora wan

int main() {
    D_LOG("=============== INICIO ===============\r\n");

    Control* control = ctrl.getControler();
    control->initialize();

    ////////////////////////////////////////////////////////////////////////// 
    // 1 - Inicializa o LoRaWAN 
    inicializa_lorawan(control);

    // 2 - Prepara as threads de Temperatura e Gyroscopio 
    // 2.1 -> A cada 20 Segundos, faz uma leitura de Temperatura e Envia pro LoRaWan 
    Thread eventTemperature(osPriorityNormal);
    eventTemperature.start(callback(read_send_temperature, control));

    // 2.2 -> Se houver 10 Mudanças no Gyroscopio, ele Aciona o GPS, que fica enviando ao LoRaWan sua localização a cada 10 segundos !!
    Thread eventTGyroscope(osPriorityNormal);
    eventTGyroscope.start(callback(read_send_gyroscope, control));

    //////////////////////////////////////////////////////////////////////////
    
    // make your event queue dispatching events forever
    wait(osWaitForever); // Para a thread principal aqui ! 

    control->finalize();
    D_LOG("=============== FINAL =============== \r\n");
    return 1;
}

int inicializa_lorawan(Control* control) {
    D_LOG("=============== LoRaWAN Comunication ===============\r\n");

    comm = (Communication<LoraData>*)control->getCommunication("LoRAWAN");

    ////////////////////////////////////////////////////////////////////////////////////////////
    // 1 - Configura os callbacks do loran WAN e inicializa os dados !!
    // Dados de Conexão
    lorawan_connect_t connect_params;
    connect_params.connect_type = LORAWAN_CONNECTION_OTAA;

    connect_params.connection_u.otaa.dev_eui = LORAWAN_DEV_EUI;
    connect_params.connection_u.otaa.app_eui = LORAWAN_APP_EUI;
    connect_params.connection_u.otaa.app_key = LORAWAN_APP_KEY;
    connect_params.connection_u.otaa.nb_trials = 10;

    LoraData* data = new LoraData(&connect_params, (LoRaRadio*)&radio, &ev_queue);
    callbacks.events = mbed::callback(lora_event_handler);
    data->prepareCallBack(callbacks);

    // Flags de leitura !!
    data->lora_port = MBED_CONF_LORA_APP_PORT;

    /////////////////////////////////////////////////////////////////////////////////////////////
    // 2 - INICIALIZA A CONEXAO
    if( comm->initialize(data) != TRUE) {
        D_LOG("Inicialização falhou !!\r\n");
        return -1;
    }

    ////////////////////////////////////////////////////////////////////////////////////////////
    // 3 - Tentar conectar 
    lorawan_status_t retcode = (lorawan_status_t) comm->connect();
    if (retcode == LORAWAN_STATUS_OK || retcode == LORAWAN_STATUS_CONNECT_IN_PROGRESS) {
        printf("Connection - In Progress ...\r\n");
    } else {
        printf("Connection error, code = %d \r\n", retcode);
        return -1;
    }
    
    return 1;
}

// --------------------------------------------------------------------------------- //
void read_send_temperature(void const *self_context) {
    D_LOG("=============== TEMPERATURE ===============\r\n");
    Control* control = (Control*)self_context;

    Sensor<void*>* temp = control->getSensor("Temperature");
    Communication<void*>* COMM = control->getCommunication("LoRAWAN");

    CayenneLPP payload(50);

    while(1) {
        float* valor = (float*)temp->getValue();
        D_LOG("TEMPERATURA -> %f\r\n", *valor);

        payload.addTemperature(1, *valor);

        // FINALIZE SENDING ...
        finalize_sending(COMM, &payload);
        
        ThisThread::sleep_for(20* ONE_SECOND); // Espera 20 Segundos !!
    }
}

void read_send_gyroscope(void const *self_context) {
    D_LOG("=============== GYROSCOPE ===============\r\n");
    Control* control = (Control*)self_context;

    Sensor<void*>* gyro = control->getSensor("Gyroscope");
    Communication<void*>* COMM = control->getCommunication("LoRAWAN");
    
    CayenneLPP payload(100);

    GyroscopeData* dataAnterior = NULL;
    int repeticoes = 0;
    bool iniciaGPS = false;

    while(1) {
        GyroscopeData* data = (GyroscopeData*) gyro->getValue();
        D_LOG("Gyro (X) => %d \r\n", data->gx);
        D_LOG("Gyro (Y) => %d \r\n", data->gy);
        D_LOG("Gyro (Z) => %d \r\n", data->gz);
        D_LOG("***********************************\r\n\r\n");

        payload.addGyrometer(1, data->gx, data->gy, data->gz);

        // FINALIZE SENDING ...
        finalize_sending(COMM, &payload);

        if(dataAnterior != NULL && iniciaGPS == false) {
            if( dataAnterior->gx != data->gx || 
                dataAnterior->gy != data->gy || 
                dataAnterior->gz != data->gz ){
                    if(++repeticoes > 10) {                        
                        iniciaGPS = true;
                        inicializa_thread_GPS(control);
                    }
            }
        }
        dataAnterior = data;
        
        ThisThread::sleep_for(10* ONE_SECOND); // Espera 10 Segundos para proxima leitura !!
    }
}

void read_send_GPS(void const *self_context) {
    D_LOG("=================== GPS ==================\r\n");
    Control* control = (Control*)self_context;

    Sensor<void*>* gps = control->getSensor("GPS");
    Communication<void*>* COMM = control->getCommunication("LoRAWAN");
    
    CayenneLPP payload(100);

    while(1) {
        GPSData* data = (GPSData*) gps->getValue();
        D_LOG("GPS (Lat) => %d \r\n", data->latitude);
        D_LOG("GPS (Long) => %d \r\n", data->longitude);
        D_LOG("GPS (Meters) => %d \r\n", data->meters);
        D_LOG("***********************************\r\n\r\n");

        payload.addGPS(1, data->latitude, data->longitude, data->meters);

        // FINALIZE SENDING ...
        finalize_sending(COMM, &payload);
        
        ThisThread::sleep_for(5* ONE_SECOND); // Espera 5 Segundos para proxima leitura !!
    }
}

void inicializa_thread_GPS(Control* control) {
    // 2.2 -> Se houver 10 Mudanças no Gyroscopio, ele Aciona o GPS, que fica enviando ao LoRaWan sua localização a cada 10 segundos !!
    Thread eventTGPS(osPriorityNormal);
    eventTGPS.start(callback(read_send_GPS, control));
}

// --------------------------------------------------------------------------------- //
void finalize_sending(Communication<void*>* COMM, CayenneLPP* payload) {
    LoraData* data = (LoraData*) COMM->getData();
    data->read_write_flags = MSG_UNCONFIRMED_FLAG;

    int16_t retcode = COMM->write(payload->getBuffer(), payload->getSize());
    D_LOG("lorawan.send = retcode [%d]\n", retcode);

    if (retcode < 0) {
        retcode == LORAWAN_STATUS_WOULD_BLOCK 
                        ? D_LOG("send - Duty cycle violation\r\n")
                        : D_LOG("send() - Error code %d \r\n", retcode);
                
        if (retcode == LORAWAN_STATUS_NO_ACTIVE_SESSIONS)
            D_LOG("\r\n|-1017 - LORAWAN_STATUS_NO_ACTIVE_SESSIONS"); 

        if (retcode == LORAWAN_STATUS_WOULD_BLOCK) { // Retry in 3 seconds
            // queue->call_in(3000, send_message); 
            D_LOG("RETRY FOI REMOVIDO DO LORAWAN !!\r\n");            
        } else {
            // queue->call_in(TX_INTERVAL, send_message);
            D_LOG("Nao re-envia a mensagem !!");
        }

        return;
    }

    // TODO: precisa re-enviar ?? queue->call_in(TX_INTERVAL, send_message); [Passar comunicacao/payload denovo ??]

    receive_message();
    D_LOG("%d bytes scheduled for transmission \r\n", retcode);
}

/**
 * Receive a message from the Network Server
 */
static void receive_message()
{
    printf("receive_message()\n");
    
    LoraData* data = (LoraData*) comm->getData();
    data->read_write_flags = MSG_CONFIRMED_FLAG|MSG_UNCONFIRMED_FLAG;

    uint8_t rx_buffer[50] = { 0 };
    int16_t retcode = comm->read( rx_buffer, sizeof(rx_buffer) ); 
    if (retcode < 0) {
        printf("receive() - Error code %d \r\n", retcode);
        return;
    }

    printf("RX Data (%d bytes): ", retcode);
    for (uint8_t i = 0; i < retcode; i++) {
        printf("%02x ", rx_buffer[i]);
    }

    printf("\r\n");
}

/**
 * Sends a message to the Network Server
 */
static void send_message() {
    printf("send_message()\n");
        
    // YOUR CODE HERE
    int16_t temperature = 10;
    printf("temperature = (%d)\n", temperature);

    CayenneLPP payload(50);
    payload.addTemperature(1, temperature);

    LoraData* data = (LoraData*) comm->getData();
    data->read_write_flags = MSG_UNCONFIRMED_FLAG;

    int16_t retcode = comm->write(payload.getBuffer(), payload.getSize());
    printf("lorawan.send = retcode [%d]\n",retcode);

    if (retcode < 0) {
        retcode == LORAWAN_STATUS_WOULD_BLOCK 
                        ? printf("send - Duty cycle violation\r\n")
                        : printf("send() - Error code %d \r\n", retcode);
                
        if (retcode == LORAWAN_STATUS_NO_ACTIVE_SESSIONS)
            printf("\r\n|-1017 - LORAWAN_STATUS_NO_ACTIVE_SESSIONS"); 

        if (retcode == LORAWAN_STATUS_WOULD_BLOCK) { //retry in 3 seconds
            ev_queue.call_in(3000, send_message);
        } else {
            ev_queue.call_in(TX_INTERVAL, send_message);
        }

        return;
    }

    ev_queue.call_in(TX_INTERVAL, send_message);

    receive_message();
    printf("%d bytes scheduled for transmission \r\n", retcode);
}

/**
 * Event handler
 */
static void lora_event_handler(lorawan_event_t event) {
    switch (event) {
        case CONNECTED:
            printf("# Connection - Successful \r\n");            
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            } else {
                ev_queue.call_in(TX_INTERVAL, send_message);
            }
            break;

        case DISCONNECTED:
            ev_queue.break_dispatch();
            printf("# Disconnected Successfully \r\n");
            break;
        case RX_DONE:
            printf("# Received message from Network Server \r\n");
            receive_message();
            break;
        case RX_TIMEOUT:
        case RX_ERROR:
            printf("# Error in reception - Code = %d \r\n", event);
            break;
        case TX_DONE:    
            count_message++;        
            printf("# Message Sent to Network Server - Count [%d] \r\n", count_message);
            break;
        case TX_TIMEOUT:
        case TX_ERROR:
        case TX_CRYPTO_ERROR:
        case TX_SCHEDULING_ERROR:
            printf("# Transmission Error - EventCode = %d \r\n", event);
            break;        
        case JOIN_FAILURE:
            printf("# OTAA Failed - Check Keys \r\n");
            break;
        case UPLINK_REQUIRED:
            printf("# Uplink required by NS \r\n");
            send_message();
            break;
        default:
            printf("# Unknown Event \r\n");
            // MBED_ASSERT("# Unknown Event");
    }
}
