#include "mbed.h"
#include "Temperature.h"
#include "Gyroscope.h"

#define ONE_SEC         1000

Serial pc(USBTX, USBRX);        // SAIDA SERIAL PADRÃO

Temperature temp(A1); // Instancia de Temperatura

I2C    i2c(I2C_SDA, I2C_SCL);       // PORTA I2C
Gyroscope giro(i2c);                // Instancia do Giroscopio
GyroscopeData* data_anterior; //Guarda a ultima leitura apra ter uma base das mudanças !!
int mudancas = 0;

// LEDs para serem acesas 
DigitalOut led1 (LED1);         // LED 1 de 3 DO BOARD
DigitalOut led2 (LED2);         // LED 2 de 3 DO BOARD
DigitalOut led3 (LED3);         // LED 3 de 3 DO BOARD
DigitalOut led_dig(D6);         // LED 1 da porta Digital 6 (A placa tem D6 e D8)

#define LOG(args...)    pc.printf(args)

// --------------------------------------------------------------------------------- //
// HEADERS  das funcoes !!
void sem_movimento_leds();
void sinaliza_led(int num_led);
bool comparaPosicao(int16_t data, int16_t data_ant, int16_t dif);

void recebeTemperatura(float* temperatura);
void fazLeituraGyroscopio(GyroscopeData* data);

// METODO PRINCIPAL !!
// --------------------------------------------------------------------------------- //
int main() {
    LOG("!!!INICIO!!!\r\n");
    sem_movimento_leds();

    //////////////////
    // 1 -> Fazer um temporizador que fica lendo a temperatura a cada 5 segundos !!
    // Chama o callback a cada 4 SEGUNDOS !
    temp.setCallbackReadOcurred(&recebeTemperatura, 4*ONE_SEC);

    //////////////////
    // 2 - Verificar se houve alguma mudança no Giroscópio
    giro.setCallbackReadOcurred(&fazLeituraGyroscopio, ONE_SEC); // leituras a cada um segundo !

    //////////////////
    // 3 - INICIALIZA AS TASKS ASSINCRONAS !!
    giro.initializeTask();
    temp.initializeTask();

    LOG("!!!FIM DE INICIALIZACAO!!!\r\n");
    int count = 0;
    while(count++ < 6) wait(10); // WAIT ONE MINUTE

    temp.~Temperature();
    giro.~Gyroscope();
    pc.printf("!!! ----------- TEMINADO -----------!!!\r\n");

    // APAGANDO LEDS !
    led_dig = 0;
    sem_movimento_leds();
}

// Outros metodos !!
// --------------------------------------------------------------------------------- //
// Pisca os led para informar q está sem movimento !!
void sem_movimento_leds() {
    int count = 0;
    while(count++ >= 2) {
        led1 = 0;
        led2 = 0;
        led3 = 0;
        ThisThread::sleep_for(1000);
        led1 = 1;
        led2 = 1;
        led3 = 1;
        ThisThread::sleep_for(1500);     
    }
    led1 = 0;
    led2 = 0;
    led3 = 0;
}

// Vai ligar o LED conforme o que tiver mudado
void sinaliza_led(int num_led) {
    led1 = 0;
    led2 = 0;
    led3 = 0;
    switch(num_led) {
        case 1: led1 = 1; break;
        case 2: led2 = 1; break;
        case 3: led3 = 1; break;
        default:
            break;
    }
}

// faz a comparação para saber se estar dentro dos parâmetros esperados !!
bool comparaPosicao(int16_t data, int16_t data_ant, int16_t dif) { 
    float menor = 0;
    float maior = 0;
    if(data_ant < 0) {
        maior = data_ant + (dif*-1);
        menor = data_ant - (dif*-1);
    } else {
        maior = data_ant + dif;
        menor = data_ant - dif;
    }

    if( data > maior || data < menor ) {
        return true; // Se a data estiver fora em algum dos limites percentuais
    }

    return false;
}

void fazLeituraGyroscopio(GyroscopeData* data) {
    int mud = 0;
    if(data_anterior == NULL) {
        data_anterior = data;
        mud = 0;
    } else if( comparaPosicao(data->gz, data_anterior->gz, 300) ) {
        data_anterior = data;
        mud = 3;
    } else if( comparaPosicao(data->gy, data_anterior->gy, 300)) {
        data_anterior = data;
        mud = 2;
    } else if( comparaPosicao(data->gx, data_anterior->gx, 800) ) {
        data_anterior = data;
        mud = 1;
    }  else {
        // SEM MOVIMENTACAO ...
    }

    switch(mud) {
        default:
        case 0: // sem mudanças (imprime somente uma vez a cada 10 vezes sem mudanças consecutivas !!)
            if( mudancas++ >= 10 ) {
                LOG("Sem movimento! \r\n");
                mudancas = 0;
                sem_movimento_leds();
            }
            break;
        case 1: // mudou eixo X
            LOG("Movimentou X .... gy_x = %d\r\n", data->gx);            
            break;
        case 2: // Mudou eixo Y
            LOG("Movimentou Y .... gy_y = %d\r\n", data->gy);
            break;
        case 3: // Mudou eixo Z
            LOG("Movimentou Z .... gy_z = %d\r\n", data->gz);            
            break;
    }

    if(mud != 0)
        mudancas = 0;

    sinaliza_led(mud);

    // LOG("*** GIRO -> (g_x = %03d; g_y = %03d; g_z = %03d)  ***\r\n", data->gx, data->gy, data->gz);
}

void recebeTemperatura(float* temperatura) {
    LOG("** Temperatura RECEBIDA => %02.02f\r\n", *temperatura );
    if(*temperatura > 30) { // ASCENDER LED !!
        LOG("** Temperatura ACIMA DO PERMITIDO !!\r\n");
        led_dig = 1;
    } else {    // APAGA O LED 
        led_dig = 0;
    }
}
