#ifndef Ambient_h
#define Ambient_h

#include "mbed.h"
#include "TCPSocketConnection.h"

#define AMBIENT_WRITEKEY_SIZE 18
#define AMBIENT_MAX_RETRY 5
#define AMBIENT_DATA_SIZE 24
#define AMBIENT_NUM_PARAMS 11
#define AMBIENT_TIMEOUT 3000 // milliseconds

/** Ambient class
 * to send data to Ambient service.
 *
 * Exsample:
 * @code
 * #include "mbed.h"
 * #include "EthernetInterface.h"
 * #include "Ambient.h"
 * #include "HDC1000.h"
 * 
 * unsigned int channelId = 100;
 * const char* writeKey = "ライトキー";
 * Ambient ambient;
 * 
 * HDC1000      hdc1000(p9,p10);
 * 
 * int main() {
 *     printf("start\r\n");
 * 
 *     EthernetInterface eth;
 *     eth.init(); //Use DHCP
 *     eth.connect();
 * 
 *     TCPSocketConnection socket;
 *     ambient.init(channelId, writeKey, &socket);
 * 
 *     printf("Ambient send to ch: %d\r\n", channelId);
 * 
 *     while (true) {
 *         float temp, humid;
 *         char tempbuf[12], humidbuf[12];
 * 
 *         hdc1000.get();
 *         temp = hdc1000.temperature();
 *         humid = hdc1000.humidity();
 * 
 *         sprintf(tempbuf, "%2.1f", temp);
 *         ambient.set(1, tempbuf);
 *         sprintf(humidbuf, "%2.0f", humid);
 *         ambient.set(2, humidbuf);
 *         printf("Temp: %s C, Humid: %s %%\r\n", tempbuf, humidbuf);
 *         
 *         ambient.send();
 *         
 *         wait(30.0);
 *     }
 * }
 * @endcode
 */
class Ambient
{
public:
    /** Create Ambient instance
     */
    Ambient(void);

    /** Initialize the instance
     * @param channelId Initialize the Ambient instance with channelId.
     * @param writeKey and writeKey
     * @param s and pointer to socket
     * @returns
     *    true on success,
     *    false on failure
     */
    bool init(unsigned int channelId, const char * writeKey, TCPSocketConnection * s, int dev = 0);

    /** Set data on field-th field of payload.
     * @param field index of payload (1 to 8)
     * @param data data
     * @returns
     *    true on success,
     *    false on failure
     */
    bool set(int field, char * data);

    /** Set data on field-th field of payload.
     * @param field index of payload (1 to 8)
     * @param data data
     * @returns
     *    true on success,
     *    false on failure
     */
    bool set(int field, int data);

    /** Set data on field-th field of payload.
     * @param field index of payload (1 to 8)
     * @param data data
     * @returns
     *    true on success,
     *    false on failure
     */
    bool set(int field, double data);

    /** Clear data on field-th field of payload.
     * @param field index of payload (1 to 8)
     * @returns
     *    true on success,
     *    false on failure
     */
    bool clear(int field);

    /** Send data to Ambient
     * @returns
     *    true on success,
     *    false on failure
     */
    bool send(void);

    /** Send bulk data to Ambient
     * @param buf pointer to the data buffer
     * @returns
     *    the number of written bytes on success (>=0)
     *    -1 on failure
     */
    int bulk_send(char * buf);

    /* Delete data stored in this channel
     * @param userKey userKey
     * @returns
     *    true on success,
     *    false on failure
     */
    bool delete_data(const char * userKey);

private:

    TCPSocketConnection * s;
    unsigned int channelId;
    char writeKey[AMBIENT_WRITEKEY_SIZE];
    int dev;
    char host[18];
    int port;

    struct {
        int set;
        char item[AMBIENT_DATA_SIZE];
    } data[AMBIENT_NUM_PARAMS];
};

#endif // Ambient_h
