#include "ld3320.h"

extern Serial &pc;
bool VoiceRecognition::SPI_inited = false;
uint8_t VoiceRecognition::n_instance = 0;

#define HIGH 1
#define LOW 0
//#define DBG_PRINT(...) pc.printf(##__VA_ARGS__)
#define DBG_PRINT(...) do{}while(0)

VoiceRecognition::VoiceRecognition(PinName cs, PinName miso, PinName mosi,
                                   PinName sck, PinName rst)
    : RSTB(rst) , CS(cs), SPI_MOSI_PIN(mosi), SPI_MISO_PIN(miso),
      SPI_SCK_PIN(sck) {
  MIC_VOL = 0x10;           // ADC增益初始值
  speech_endpoint = 0x15;   //语音端点检测初始值
  speech_start_time = 0x08; //语音端点检测开始时间初始值
  speech_end_time = 50;     //语音端点检测结束时间初始值
  voice_max_length = 0x3c;  //最长语音段时间，默认6秒
  noise_time = 0x02;        //忽略上电噪声时间
  asr_state = LD_ASR_NONE;
  readnum = 0;
  instance_id = ++n_instance;
  for (int i = 0; i < sizeof(asr_cmd_str) / sizeof(asr_cmd_str[0]); i++)
    asr_cmd_str[i] = NULL;
}

int VoiceRecognition::read()//识别结果读取
{
    int ret = -1;
    if (asr_state == LD_ASR_NONE) {
        restart();
    }
    if (readReg(0x2b) & 0x1c) { //模拟中断调用
        update();
    }
    if (asr_state == LD_ASR_FOUNDOK) {
        ret = readnum;
        asr_state = LD_ASR_NONE;
        // restart();
    } else if (asr_state == LD_ASR_FOUNDZERO) {
        asr_state = LD_ASR_NONE;
        // restart();
    } else if (asr_state == LD_ASR_ERROR) {
        DBG_PRINT("resume from error\n");
        asr_state = LD_ASR_NONE;
        // restart();
    }
    return ret;
}
void VoiceRecognition::restart()
{
    reset();
    ASR_init();
    start();
    return;
}
void VoiceRecognition::update()//中断服务函数
{
    uint8_t Asr_Count = 0;
    uint8_t reg2b = readReg(0x2b);
    readReg(0x29);
    writeReg(0x29, 0) ; ///////////关中断
    readReg(0x02);
    writeReg(0x02, 0) ; /////////////关FIFO中断
    if ((reg2b & 0x10) && readReg(0xb2) == 0x21 && readReg(0xbf) == 0x35) { //如果有语音识别中断、DSP闲、ASR正常结束
        Asr_Count = readReg(0xba);//读中断辅助信息
        if (Asr_Count > 0 && Asr_Count < 4) { //////如果有识别结果
            DBG_PRINT("ASR ok\n");
            readnum = readReg(0xc5);
            asr_state = LD_ASR_FOUNDOK;
        } else {
            DBG_PRINT("ASR 0 result\n");
            asr_state = LD_ASR_FOUNDZERO;
        }
        writeReg(0x2b, 0); //////清楚中断编号
        writeReg(0x1C, 0); ////////貌似关麦克风啊~~为毛
    } else if ((reg2b & 0x08)) {
        DBG_PRINT("interal error\n");
        asr_state = LD_ASR_ERROR;
    }
    // static uint8_t dbg;
    // DBG_PRINT(instance_id, HEX);
    // DBG_PRINT(" update=");
    // DBG_PRINT(dbg++, HEX);
    // DBG_PRINT('\n');
}
void VoiceRecognition::cSHigh()  //CS拉高
{
    CS = HIGH;
}
void VoiceRecognition::cSLow()  //CS脚拉低
{
    CS = LOW;
}
void VoiceRecognition::spiDelay(int _time)
{
    _time *= 4;
    for (volatile int delay = _time; delay--;);
}
void VoiceRecognition::writeReg(unsigned char address, unsigned char value) ////////写寄存器，参数（寄存器地址，数据）
{
    cSLow();////拉低CS
    spiDelay(60);
    transfer(0x04);////////////写指令
    spiDelay(60);
    transfer(address);
    spiDelay(60);
    transfer(value);
    spiDelay(60);
    cSHigh();////拉高CS
}

unsigned char VoiceRecognition::readReg(unsigned char address)///读寄存器，参数（寄存器地址）
{
    unsigned char result;
    cSLow();////拉低CS
    spiDelay(60);
    transfer(0x05);///////////读指令
    spiDelay(60);
    transfer(address);
    spiDelay(60);
    result = transfer(0x00);
    spiDelay(60);
    cSHigh();///拉高CS
    spiDelay(60);
    return (result);
}
uint8_t VoiceRecognition::transfer(uint8_t _data) /////////////////SPI数据交换
{
#ifndef SOFTWARE_SPI
    SPDR = _data;
    while (!(SPSR & _BV(SPIF)));
    return SPDR;
#else
    uint8_t ret = 0;
    for (uint8_t bit = 0; bit < 8; bit++) {
        spiDelay(10);;
        ret <<= 1;
        ret |= (SPI_MISO_PIN);
        SPI_MOSI_PIN = (_data & 0x80) ? HIGH : LOW;
        _data <<= 1;
        spiDelay(50);;
        SPI_SCK_PIN = LOW;
        spiDelay(60);
        SPI_SCK_PIN = HIGH;
    }
    return ret;
#endif
}

void VoiceRecognition::init(uint8_t mic)////////模块启用，参数为麦克风选择（MIC/MONO）与丝印对照,在SETUP中调用
{
    if (mic == MIC) {
        g_Mic = MIC;
    } else if (mic == MONO) {
        g_Mic = MONO;
    }
    cSHigh();

    SPI_SCK_PIN = HIGH;
    if (!VoiceRecognition::SPI_inited) {

#ifndef SOFTWARE_SPI
        // SS must be in output mode even it is not chip select
        pinMode(LD_CHIP_SELECT_PIN, OUTPUT);
        LD_CHIP_SELECT_PIN = HIGH); // disable any SPI device using hardware SS 拉高;
        // Enable SPI, Master, clock rate f_osc/128
        SPCR = (1 << SPE) | (1 << MSTR) | (1 << SPR1) | (1 << SPR0);/////初始化SPI寄存器
        // clear double speed
        SPSR &= ~(1 << SPI2X);//2倍速
        SPCR = (SPCR & ~SPI_MODE_MASK) | 0x08;//设置SCK常态电平与取样时间，0x08为SCK常态为高电平，下降沿有效
#endif  // SOFTWARE_SPI 


        VoiceRecognition::SPI_inited = true;
    }
    reset();//LD3320复位操作

    // #if defined(__AVR_ATmega32U4__)
    //  attachInterrupt(1,update,LOW);//开中断
    // #else
    //  attachInterrupt(0,update,LOW);//开中断
    // #endif

    ASR_init();///语音识别初始化函数
}

void VoiceRecognition::reset()//LD3320复位操作
{
	  DBG_PRINT("ld3320 reset\n");
    RSTB = HIGH;
    wait_ms(1);
    RSTB = LOW;
    wait_ms(5);
    RSTB = HIGH;
    wait_ms(50);
    // cSLow();
    // wait_ms(1);
    // cSHigh();
    // wait_ms(1);
    readReg(0x06);
    if (readReg(0x06) != 0x87)
        goto fail_reg1;
    if (readReg(0x35) != 0x80)
        goto fail_reg2;
    if (readReg(0xb3) != 0xff)
        goto fail_reg3;
    /*
    DBG_PRINT("reg init:");
    DBG_PRINT(readReg(0x06), HEX);
    DBG_PRINT(' ');
    DBG_PRINT(readReg(0x06), HEX);
    DBG_PRINT(' ');
    DBG_PRINT(readReg(0x35), HEX);
    DBG_PRINT(' ');
    DBG_PRINT(readReg(0xb3), HEX);
    DBG_PRINT('\n');

    DBG_PRINT("reg R/W test: %#x", readReg(0x06));
    writeReg(0x35, 0x33);
    writeReg(0x1b, 0x55);
    writeReg(0xb3, 0xaa);
    DBG_PRINT(" %#x %#x %#x\n", 
			readReg(0x35), readReg(0x1b), readReg(0xb3));
    // writeReg(0xb9, 0x00);
    // DBG_PRINT(instance_id);
    // DBG_PRINT(" reg init\n");
    */
    return;
fail_reg1:
fail_reg2:
fail_reg3:
    DBG_PRINT("%d reg init failed\n", instance_id);
    asr_state = LD_ASR_ERROR;
    return;
}

void VoiceRecognition::ASR_init()////////////初始化语音识别模式、
{
    //添加状态标记
    readReg(0x06);
    writeReg(0x17, 0x35);
    wait_ms(10);
    readReg(0x06);
    writeReg(0x89, 0x03);
    wait_ms(5);
    writeReg(0xcf, 0x43);
    wait_ms(5);
    writeReg(0xcb, 0x02);
    writeReg(0x11, PLL_11);
    writeReg(0x1e, 0x00);
    writeReg(0x19, PLL_ASR_19);
    writeReg(0x1b, PLL_ASR_1B);
    writeReg(0x1d, PLL_ASR_1D);
    wait_ms(10);
    writeReg(0xcd, 0x04);
    writeReg(0x17, 0x4c);
    wait_ms(5);
    writeReg(0xb9, 0x00);
    writeReg(0xcf, 0x4f);
    writeReg(0x6f, 0xff);

    writeReg(0xbd, 0x00);
    writeReg(0x17, 0x48);
    wait_ms(10);
    writeReg(0x3c, 0x80);
    writeReg(0x3e, 0x07);
    writeReg(0x38, 0xff);
    writeReg(0x3a, 0x07);
    writeReg(0x40, 0);
    writeReg(0x42, 8);
    writeReg(0x44, 0);
    writeReg(0x46, 8);
    wait_ms(1);

    // DBG_PRINT("test: ");
    // DBG_PRINT(readReg(0x19), HEX);
    // DBG_PRINT("\n");
}
void VoiceRecognition::addCommand(const char *pass, int num) {
  for (int i = 0; i < sizeof(asr_cmd_str) / sizeof(asr_cmd_str[0]); i++)
    if (asr_cmd_str[i] == NULL) {
      asr_cmd_str[i] = pass;
      asr_cmd_number[i] = num;
      break;
    }
}
void VoiceRecognition::addAllCommands()
{
    for (int i = 0; i < sizeof(asr_cmd_str) / sizeof(asr_cmd_str[0]); i++) {
        if (asr_cmd_str[i] == NULL) {
            break;
        }
        doAddCommand(asr_cmd_str[i], asr_cmd_number[i]);
    }
}

void VoiceRecognition::doAddCommand(const char *pass, int num)
{

    int i;
    if (check_b2() == 0) {
        DBG_PRINT("ASR Busy, addCommand failed\n");
        asr_state = LD_ASR_ERROR;
        return;
    }
    writeReg(0xc1, num);//字符编号
    writeReg(0xc3, 0 );//添加时输入00
    writeReg(0x08, 0x04);//不清除

    wait_ms(1);
    writeReg(0x08, 0x00);//
    wait_ms(1);
    for (i = 0; i <= 80; i++) {
        if (pass[i] == 0)
            break;
        writeReg(0x5, pass[i]);///写入FIFO_EXT
    }
    writeReg(0xb9, i);//写入当前添加字符串长度
    writeReg(0xb2, 0xff);//////////B2全写ff
    writeReg(0x37, 0x04);//添加语句
    if (readReg(0xbf) != 0x31) {
        DBG_PRINT("addCommand error\n");
        asr_state = LD_ASR_ERROR;
    }
}
unsigned char VoiceRecognition::start()//////开始识别
{
    addAllCommands();
    if (asr_state == LD_ASR_ERROR)
        return 0;

    writeReg(0x35, MIC_VOL);////adc增益；会影响识别范围即噪声

    writeReg(0xb3, speech_endpoint);//语音端点检测控制

    writeReg(0xb4, speech_start_time);//语音端点起始时间

    writeReg(0xb5, speech_end_time);//语音结束时间

    writeReg(0xb6, voice_max_length);//语音结束时间

    writeReg(0xb7, noise_time);//噪声时间

    writeReg(0xb8, 10);//识别超时时间，单位秒

    writeReg(0x1c, 0x09);////////麦克风设置保留

    writeReg(0xbd, 0x20);/////////保留设置
    writeReg(0x08, 0x01);///////////→清除FIFO_DATA
    wait_ms( 1);
    writeReg(0x08, 0x00);////////////清除指定FIFO后再写入一次00H
    wait_ms( 1);
    if (check_b2() == 0) { ////////读取0xB2寄存器函数如果DSP没在闲状态则RETURN 0
        DBG_PRINT("ASR_run failed\n");
        asr_state = LD_ASR_ERROR;
        return 0;
    }
    writeReg(0xb2, 0xff);////////给0xB2写FF

    writeReg(0x37, 0x06);////////开始识别
    wait_ms( 5 );
    if (readReg(0xbf) < 0x32) {
        DBG_PRINT("ASR_run error\n");
        asr_state = LD_ASR_ERROR;
        return 0;
    }
    writeReg(0x1c, g_Mic);////////选择麦克风
    writeReg(0x29, 0x10);////////开同步中断
    writeReg(0xbd, 0x00);/////////启动为语音识别

    asr_state = LD_ASR_RUNING;

    return 1;////返回1
}

int VoiceRecognition::check_b2()////////用作检测芯片工作是否正常，或者DSP是否忙，不需用户操作，正常/闲返回1
{
    for (int j = 0; j < 10; j++) {
        if (readReg(0xb2) == 0x21) {
            return 1;
        }
        wait_ms(10);
    }
    return 0;
}

void VoiceRecognition::micVol(uint8_t vol)//调整ADC增益，参数（0x00~0xFF,建议10-60）；
{
    MIC_VOL = vol;
    writeReg(0x35, MIC_VOL);////adc增益；会影响识别范围即噪声
}
void VoiceRecognition::speechEndpoint(uint8_t speech_endpoint_)//调整语音端点检测，参数（0x00~0xFF,建议10-40）；
{
    speech_endpoint = speech_endpoint_;
    writeReg(0xb3, speech_endpoint);//语音端点检测控制
}

void VoiceRecognition::speechStartTime(uint8_t speech_start_time_)//调整语音端点起始时间，参数（0x00~0x30,单位10MS）；
{
    speech_start_time = speech_start_time_;
    writeReg(0xb4, speech_start_time);//语音端点起始时间
}
void VoiceRecognition::speechEndTime(uint8_t speech_end_time_)//调整语音端点结束时间（吐字间隔时间），参数（0x00~0xC3,单位10MS）；
{
    speech_end_time = speech_end_time_;
    writeReg(0xb5, speech_end_time);//语音结束时间
}
void VoiceRecognition::voiceMaxLength(uint8_t voice_max_length_)//最长语音段时间，参数（0x00~0xC3,单位100MS）；
{
    voice_max_length = voice_max_length_;
    writeReg(0xb6, voice_max_length);//语音
}
void VoiceRecognition::noiseTime(uint8_t noise_time_)//上电噪声略过，参数（0x00~0xff,单位20MS）；
{
    noise_time = noise_time_;
    writeReg(0xb7, noise_time);//噪声时间
}

