#include "mbed.h"
#include "EthernetInterface.h"

#include "FXAS21002.h"
#include "FXOS8700.h"  
 
#include "NetworkAPI/buffer.hpp"
#include "NetworkAPI/select.hpp"
#include "NetworkAPI/ip/address.hpp"
#include "NetworkAPI/tcp/socket.hpp"
using namespace network;
 
#define MAX_CLIENTS 5
#define TCP_PORT 55

// Initialize RGB LEDs
DigitalOut RGBLED1(LED1);
DigitalOut RGBLED2(LED2);
DigitalOut RGBLED3(LED3);

int initPulses = 15;

// Initialize Serial port
Serial pc(USBTX, USBRX);

// Initialize pins for I2C communication for sensors. Set jumpers J6,J7 in FRDM-STBC-AGM01 board accordingly.
FXOS8700 accel(D14,D15);
FXOS8700 mag(D14,D15);
FXAS21002 gyro(D14,D15);

float accel_data[3];
float mag_data[3];
float gyro_data[3];
 
// Ethernet initialization and connection return values
int retInit;
int retConn;

int main() {
    // Start with the LEDs off
    RGBLED1 = 1;
    RGBLED2 = 1;
    RGBLED3 = 1;

    // Configure Accelerometer FXOS8700, Magnetometer FXOS8700 & Gyroscope FXAS21002
    accel.accel_config();
    mag.mag_config();
    gyro.gyro_config();

    // Set baud rate
    pc.baud(115200);
    pc.printf("Attempting to connect to the network\r\n"); 
   
    // Intialize Ethernet connection
    EthernetInterface eth;
    
    pc.printf("Initializing connection...");   
    retInit = eth.init("10.12.1.55","255.255.0.0","10.12.0.1"); //Use a static IP address
    
    if (retInit == 0)
    {
        pc.printf("OK (IP = %s)\r\n", eth.getIPAddress());
        
        for (int a = 0; a < initPulses; a = a+1)
        { 
            RGBLED1 = 1;
            RGBLED2 = 1;
            wait(0.025);
            RGBLED1 = 0;
            RGBLED2 = 0;
            wait(0.025);
        }
        
        pc.printf("Connecting..."); 
        retConn = eth.connect();

        if (retConn == -1)
        //if (retConn == 0)
        {
            pc.printf("OK\r\n");
            pc.printf("Success! Device with IP Address %s is connected to network!\r\n", eth.getIPAddress());

            RGBLED1 = 1;            

            for (int a = 0; a < initPulses; a = a+1)
            { 
                RGBLED2 = 1;
                wait(0.025);
                RGBLED2 = 0;
                wait(0.025);
            }
        }
        else
        {
            pc.printf("FAILED\r\n");
            pc.printf("Connection failure! Device not connected to network!\r\n", eth.getIPAddress());

            RGBLED1 = 0;
        }
    }
    else
    {
        pc.printf("FAILED\r\n");
        pc.printf("Initialization failure! Device not connected to network!\r\n", eth.getIPAddress());
        
        RGBLED1 = 0;
    }
     
    Select select;
    tcp::Socket server;
    tcp::Socket client[MAX_CLIENTS];
    tcp::Socket *socket = NULL;
     
    int result = 0;
    int index = 0;

    char imuValStr [1024];
    char welcomeMsgStr [128] = "Connected to IMU";
     
    network::Buffer inbuffer(256);
    network::Buffer outbuffer(256);
     
    // Configure the server socket (assume everty thing works)
    server.open();
    server.bind(TCP_PORT);
    pc.printf("TCP Port: %d\r\n\n", TCP_PORT);
    server.listen(MAX_CLIENTS);
   
    // Add sockets to the select api
    select.set(&server, Select::Read);
    for (index = 0; index < MAX_CLIENTS; index++) {
        select.set(&client[index], Select::Read);
    }
     
    do {
        // Wait for activity
        result = select.wait();
        if (result < -1) {
            pc.printf("Failed to select\n\r");
            break;
        }
         
        // Get the first socket
        socket = (tcp::Socket *)select.getReadable();
         
        for (; socket != NULL; socket = (tcp::Socket *)select.getReadable()) {
            // Check if there was a connection request.
            if (socket->getHandle() == server.getHandle()) {                
                // Find an unused client
                for (index = 0; index < MAX_CLIENTS; index++) {
                    if (client[index].getStatus() == network::Socket::Closed) {
                        break;
                    }
                }
                 
                // Maximum connections reached
                if (index == MAX_CLIENTS) {
                    pc.printf("Maximum connections reached\n\r");
                    continue;
                }
             
                // Accept the client
                socket->accept(client[index]);
                pc.printf("Client connected %s:%d\n\r",
                    client[index].getRemoteEndpoint().getAddress().toString().c_str(),
                    client[index].getRemoteEndpoint().getPort());
                
                // Send data to the client
                client[index].write(welcomeMsgStr,strlen(welcomeMsgStr));
                continue;
            }
             
            // It was not the server socket, so it must be a client talking to us.
            switch (socket->read(inbuffer)) {
                case 0:
                    // Remote end disconnected
                    pc.printf("Client disconnected %s:%d\n\r",
                        socket->getRemoteEndpoint().getAddress().toString().c_str(),
                        socket->getRemoteEndpoint().getPort());
                     
                    // Close socket
                    socket->close();
                    break;
                 
                case -1:
                    pc.printf("Error while reading data from socket\n\r");
                    socket->close();
                    break;
                 
                default:
                    pc.printf("Message from %s:%d\n\r",
                        socket->getRemoteEndpoint().getAddress().toString().c_str(),
                        socket->getRemoteEndpoint().getPort());
                         
                    pc.printf("%s\n\r", (char *)inbuffer.data());

                    // Get IMU data
                    accel.acquire_accel_data_g(accel_data);     
                    gyro.acquire_gyro_data_dps(gyro_data);     
                    mag.acquire_mag_data_uT(mag_data);
                    
                    // Print IMU data to serial port
                    pc.printf("Message to %s:%d",
                        socket->getRemoteEndpoint().getAddress().toString().c_str(),
                        socket->getRemoteEndpoint().getPort());
                    pc.printf("ACC(g):%4.4f,%4.4f,%4.4f|GYRO(deg/s):%4.4f,%4.4f,%4.4f|MAG(uT):%4.4f,%4.4f,%4.4f\n",
                        accel_data[0],
                        accel_data[1],
                        accel_data[2],
                        gyro_data[0],
                        gyro_data[1],
                        gyro_data[2],
                        mag_data[0],
                        mag_data[1],
                        mag_data[2]); 
                    sprintf(imuValStr,"ACC(g):%4.4f,%4.4f,%4.4f|GYRO(deg/s):%4.4f,%4.4f,%4.4f|MAG(uT):%4.4f,%4.4f,%4.4f\n",
                        accel_data[0],
                        accel_data[1],
                        accel_data[2],
                        gyro_data[0],
                        gyro_data[1],
                        gyro_data[2],
                        mag_data[0],
                        mag_data[1],
                        mag_data[2]);
                    
                    // Send data to the client
                    client[index].write(imuValStr,strlen(imuValStr));
                    break;
            }
        }
             
    } while (server.getStatus() == network::Socket::Listening);
}