#include "cr_gps.h"

////////////////////////////////////////////////////
// Private                                        //
////////////////////////////////////////////////////

int CelestialReachGPS::fromHex(char a)
{
  if (a >= 'A' && a <= 'F')
    return a - 'A' + 10;
  else if (a >= 'a' && a <= 'f')
    return a - 'a' + 10;
  else
    return a - '0';
}

void CelestialReachGPS::readLatitude(int i)
{
    int deg, min, sec;
    double fsec, val;
    
    deg = ((gpsline[i] - '0') * 10) + gpsline[i+1] - '0';
    min = ((gpsline[i+2] - '0') * 10) + gpsline[i+3] - '0';
    sec = (((gpsline[i+5] - '0') * 1000) + ((gpsline[i+6] - '0') * 100) + ((gpsline[i+7] - '0') * 10) + (gpsline[i+8] - '0'));
    
    fsec = (double)((double)sec /10000.0);
    
    val = (double)deg + ((double)((double)min/60.0)) + (fsec/60.0);
    
    if (gpsline[i+10] == 'S') { val *= -1.0; }
    
    currentPosition.latitude = val;
}

void CelestialReachGPS::readLongitude(int i)
{
    int deg, min, sec;
    double fsec, val;
    
    deg  = ((gpsline[i] - '0') * 100) + ((gpsline[i+1] - '0') * 10) + (gpsline[i+2] - '0');
    min  = ((gpsline[i+3] - '0') * 10) + gpsline[i+4] - '0';
    sec  = (((gpsline[i+6] - '0') * 1000) + ((gpsline[i+7] - '0') * 100) + ((gpsline[i+8] - '0') * 10) + (gpsline[i+9] - '0'));
    
    fsec = (double)((double)sec /10000.0);
    
    val  = (double)deg + ((double)((double)min/60.0)) + (fsec/60.0);
    
    //This needs to be set to 12
    if (gpsline[i+12] == 'W') { val *= -1.0; }
    
    currentPosition.longitude = val;
}

void CelestialReachGPS::processGSV()
{
    tokenNum = 0;
        
    for (int i = 0; i < gpslinePos; i++)
    {        
        if (gpsline[i] == ',')
        {
            if (gpsline[i+1] != ',')
            {
                i++;
            }
            
            switch (tokenNum++)
            {
                case 0: // msg total
                    break;
                case 1: // msg number                    
                    break;
                case 2:
                    currentSatsInView = ((gpsline[i] - '0') * 10) + gpsline[i+1] - '0';
                    i += 1;
                    break;
                case 3: // PRN
                    //pc.printf("%d] Sat %s", sc, tokens);
                    break;
                case 4: // elevation
                    //pc.printf(" Elev: %s", tokens);
                    break;
                case 5: // azimuth
                    //pc.printf(" Az: %s\r\n", tokens);
                    break;
                case 6: // snr
                    //pc.printf(" SNR: %s\r\n", tokens);
                    
                    //if (sc++ < 3)
                    //    tokenPos = 4;
                    break;
            }
        }
    }
}


// GPS Satellites Active
void CelestialReachGPS::processGSA()
{
    float dop = 0;
    
    tokenNum = 0;
    
    for (int i = 0; i < gpslinePos; i++)
    {        
        if (gpsline[i] == ',')
        {   
            if (gpsline[i+1] != ',')
            {
                i++;
            }
                
            switch (tokenNum++)
            {
                case 0: // fix type (manual or auto... meh)
                    break;
                case 1: // Fix Mode
                    if (gpsline[i] == '1')
                    {
                        currentFix.valid = true;
                    }
                    else if (gpsline[i] == '2') // 2D
                    {
                        currentFix.type = twoD;
                    }
                    else if (gpsline[i] == '3') // 3D
                    {
                        currentFix.type = threeD;
                    }
                    
                    break;
                case 14: // PDOP
                    dop = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        dop *= 10;
                        dop += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { dop += (float)(gpsline[i+1] - '0') / (float)10.0; }
                    
                    currentFix.pdop = dop;
                    break;
                case 15: // HDOP
                    dop = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        dop *= 10;
                        dop += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { dop += (float)(gpsline[i+1] - '0') / (float)10.0; }
                    
                    currentFix.hdop = dop;
                    break;
                case 16: // VDOP
                    dop = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        dop *= 10;
                        dop += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { dop += (float)(gpsline[i+1] - '0') / (float)10.0; }
                    
                    currentFix.vdop = dop;
                    break;
            }
        }
    }
}

// Recommended Minimum Sentence
void CelestialReachGPS::processRMC()
{
    tokenNum = 0;
    
    for (int i = 0; i < gpslinePos; i++)
    {
        if (gpsline[i] == ',')
        {
            i++;
            switch (tokenNum++)
            {
                case 0: // currentTime
                {
                    currentTime.hour = ((gpsline[i] - '0') * 10) + gpsline[i+1] - '0';
                    currentTime.minute = ((gpsline[i+2] - '0') * 10) + gpsline[i+3] - '0';
                    currentTime.second = ((gpsline[i+4] - '0') * 10) + gpsline[i+5] - '0';
                    // venus outputs fractions of a second... but i dont think we need that level of detail
                    
                    i += 5;
                    
                    break;
                }
                case 1: // tracking status
                {
                    if (gpsline[i] == 'A')
                    {
                        currentFix.valid = true;
                    }
                    else
                    {
                        currentFix.valid = false;
                    }
                    
                    break;
                }
                case 2: // Latitude
                {
                    readLatitude(i);
                    
                    i += 10;
                    break;
                }
                case 3: // longitude
                {
                    readLongitude(i);
                    
                    i += 11;
                    break;
                }
                case 4: // speed in knots
                {
                    double speed = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        speed *= 10;
                        speed += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { speed += (double)(gpsline[i+1] - '0') / 10.0; }
                    
                    //todo: something is wrong with speed....
                    currentTrack.groundSpeedKnots = speed;
                    currentTrack.groundSpeedKmh = speed * 1.852;
                    
                    break;
                }
                case 5: // heading
                {
                    double trackHeading = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        trackHeading *= 10;
                        trackHeading += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { trackHeading += (double)(gpsline[i+1] - '0') / 10.0; }
                    
                    currentTrack.heading = trackHeading;
                    
                    break;
                }
                case 6: // date
                {
                    currentTime.day = ((gpsline[i] - '0') * 10) + gpsline[i+1] - '0';
                    currentTime.month = ((gpsline[i+2] - '0') * 10) + gpsline[i+3] - '0';
                    currentTime.year = ((gpsline[i+4] - '0') * 10) + gpsline[i+5] - '0'; // + 2000?
                    
                    i += 5;
                    break;
                }
            }
        }
    }
}

// Track made good
void CelestialReachGPS::processVTG()
{
    tokenNum = 0;
    
    for (int i = 0; i < gpslinePos; i++)
    {
        if (gpsline[i] == ',')
        {
            i++;
            switch (tokenNum++)
            {
                case 0: // fix time
                {
                    double trackHeading = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        trackHeading *= 10;
                        trackHeading += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { trackHeading += (double)(gpsline[i+1] - '0') / 10.0; }
                    
                    currentTrack.heading = trackHeading;
                    
                    break;
                }
                case 4: // speed knots
                {
                    double speed = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        speed *= 10;
                        speed += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { speed += (double)(gpsline[i+1] - '0') / 10.0; }
                    
                    currentTrack.groundSpeedKnots = speed;
                    
                    break;
                }
                case 6: // speed kmh
                {
                    double speed = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        speed *= 10;
                        speed += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { speed += (double)(gpsline[i+1] - '0') / 10.0; }
                    
                    //todo: something is wrong with speed....
                    currentTrack.groundSpeedKmh = speed;
                    
                    break;
                }
            }
        }
    }
}

// General Fix Data
void CelestialReachGPS::processGGA()
{
    tokenNum = 0;
    
    for (int i = 0; i < gpslinePos; i++)
    {
        if (gpsline[i] == ',')
        {
            i++;
            switch (tokenNum++)
            {
                case 0: // fix time
                {
                    currentTime.hour = ((gpsline[i] - '0') * 10) + gpsline[i+1] - '0';
                    currentTime.minute = ((gpsline[i+2] - '0') * 10) + gpsline[i+3] - '0';
                    currentTime.second = ((gpsline[i+4] - '0') * 10) + gpsline[i+5] - '0';
                    
                    i += 5;
                    break;  
                }
                case 1: // latitude
                {
                    readLatitude(i);
                    
                    i += 10;
                    break;
                }
                case 2: // longitude
                {
                    readLongitude(i);
                    
                    i += 11;
                    break;
                }
                case 3: // fix quality
                {
                    currentFixQual = gpsline[i] - '0';
                    break;
                }
                case 4: // satellites tracked
                {
                    currentSatsTracked = ((gpsline[i] - '0') * 10) + gpsline[i+1] - '0';
                    i += 1;
                    break;
                }
                case 5: // hdop
                {
                    float dop = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        dop *= 10;
                        dop += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { dop += (float)(gpsline[i+1] - '0') / (float)10.0; }
                    
                    currentFix.hdop = dop;
                    break;
                }
                case 6: // altitude
                {
                    double alt = 0;
                    while (gpsline[i] != '.' && gpsline[i] != ',')
                    {
                        alt *= 10;
                        alt += gpsline[i++] - '0';
                    }
                    if (gpsline[i] == ',') { i--; }
                    if (gpsline[i] == '.') { alt += (double)(gpsline[i+1] - '0') / 10.0; }
                    
                    currentPosition.altitude = alt;
                    break;
                }
            }
        }
    }
}

void CelestialReachGPS::irq_rx()
{
    char c;
    int read = 0;
        
    while (readable())
    {
        c = getc();
        read++;
        
        switch (state)
        {
            case gps_lookingForSentence:
                if (c == '$')
                {
                    state = gps_lookingForCS;
                    gpslinePos = 0;
                    rxcs = 0;
                    checksum = 0;
                    csA = 0;
                    csB = 0;
                    
                }
            break;
            case gps_lookingForCS:
                if (c == '$') // unexpected sentence start
                {
                    gpslinePos = 0;
                    rxcs = 0;
                    checksum = 0;
                    csA = 0;
                    csB = 0;
                }
                else if (c == '*')
                {
                    state = gps_lookingForLineEnd;
                    rxcs = 0;
                    //pc.printf("\r\nchecksum calc: %x; recieved: ", checksum);
                }
                else
                {
                    checksum ^= c;
                    gpsline[gpslinePos++] = c;
                }
            break;
            case gps_lookingForLineEnd:
                if (c == '\r' || c == '\n')
                {
                    state = gps_lookingForSentence;
                    
                    if (checksum != rxcs)
                    {
                        /*pc.printf("\r\n>");
                        for (int i = 0; i < gpslinePos; i++)
                        {
                            pc.putc(gpsline[i]);
                        }
                        pc.printf("      << (GPSLINE)");
                        pc.printf("\r\nchecksum error: %02x; rxcs: %02x CS a[%c] b[%c]\r\n\r\n", checksum, rxcs, csA, csB);
                        */
                        checkfail++;
                        
                        break;
                    }
                    
                    // get the first 3 characters of the sentence type.
                    // ublox and other GNSS systems will use GN/GL/GB/GP depending on the source of the content for the first two characters
                    // we dont care what satellite system the data came from.
                    char sentenceType[3];
                    for (int i = 0; i < 3; i++)
                    {
                        sentenceType[i] = gpsline[i + 2];
                    }
                    
                    if (strncmp(sentenceType, GGA_string, 3) == 0)
                    {
                        packetType = gps_gga_packet;
                        
                        processGGA();
                    }
                    else if (strncmp(sentenceType, GSV_string, 3) == 0)
                    {
                        packetType = gps_gsv_packet;
                        
                        processGSV();
                    }
                    else if (strncmp(sentenceType, VTG_string, 3) == 0)
                    {
                        packetType = gps_vtg_packet;
                        
                        processVTG();
                    }
                    else if (strncmp(sentenceType, GSA_string, 3) == 0)
                    {
                        packetType = gps_gsa_packet;
                        
                        processGSA();
                    }
                    else if (strncmp(sentenceType, RMC_string, 3) == 0)
                    {
                        packetType = gps_rmc_packet;
                        
                        processRMC();
                    }
                    else
                    {
                        packetType = 0;
                    }
                    
                    hasReceivedData = true;
                }
                else
                {
                    if (csA == 0)
                    {
                        csA = c;
                        rxcs = fromHex(c) * 16;
                    }
                    else
                    {
                        csB = c;
                        
                        rxcs += fromHex(c);
                    }
                }
            break;
        }
        
        // it appears to be much more efficient to just stay in the interrupt looking for the next byte than interrupt for every byte.
        if (state != gps_lookingForSentence)
        {
            int cnt = 0;
            while (!readable() && cnt++ < 100)
            {
                wait(baudwait);
            }
        }
    }
}


// my venus doesnt want to respond to anything :(
void CelestialReachGPS::VenusWriteData(uint8_t *data, int length, int messageId)
{
    int cs = messageId;
    
    putc(0xA0);
    putc(0xA1);
    
    putc(length >> 8);
    putc(length & 0xFF);
    putc(messageId);
    
    for (int i = 0; i < length - 1; i++)
    {
        putc((int)data[i]);
        cs ^= (int)data[i];
    }
    
    putc(cs);
    putc(0x0D);
    putc(0x0A);
}


////////////////////////////////////////////////////
// Public                                         //
////////////////////////////////////////////////////


void CelestialReachGPS::VenusGpsConfig()
{
    uint8_t payloadData[3];
    int payloadLength = 0;
    int messageId = 0;
    
    messageId = 0x02;
    payloadLength = 2;
    payloadData[0] = 00;
    VenusWriteData(payloadData, payloadLength, messageId);
    
    //wait(0.05);
    messageId = 0x05; // Configure Serial Port
    payloadLength = 4; // length includes message id
    payloadData[0] = 0; // uart 0
    payloadData[1] = 5; // 0 = 4800, 1 = 9600, 2 = 19200, 3 = 38400, 4 = 57600, 5 = 115200
    payloadData[2] = 0; // 0 = update to SRAM, 1 = updated to SRAM and Flash
    VenusWriteData(payloadData, payloadLength, messageId);
    baud(57600);
    
    //wait(0.05);
    messageId = 0x0E; // Configure update rate
    payloadLength = 3;
    payloadData[0] = 4; // 4 times per second
    payloadData[1] = 0; // 0 = update to SRAM, 1 = updated to SRAM and Flash
    VenusWriteData(payloadData, payloadLength, messageId);
}

gps_time_t CelestialReachGPS::lastTime()
{
    return currentTime;
}

gps_track_t CelestialReachGPS::lastTrack()
{
    return currentTrack;
}

gps_fix_t CelestialReachGPS::lastFix()
{
    return currentFix;
}
    
gps_position_t CelestialReachGPS::lastPosition()
{
    return currentPosition;
}

int CelestialReachGPS::lastFixType()
{
    return currentFixQual;
}

int CelestialReachGPS::lastSatsTracked()
{
    return currentSatsTracked;
}

int CelestialReachGPS::lastSatsInView()
{
    return currentSatsInView;
}

int CelestialReachGPS::checksumErrors()
{
    return checkfail;
}


void CelestialReachGPS::setBaud(int baudRate)
{
    baudwait = baudRate/8000000/30;
    baud(baudRate);
}


////////////////////////////////////////////////////
// Constructor                                    //
////////////////////////////////////////////////////
CelestialReachGPS::CelestialReachGPS(PinName tx, PinName rx, int baudRate) : Serial(tx, rx)
{
    baudwait = baudRate/8000000/30;
    
    gpslinePos = 0;
    
    state = gps_lookingForSentence;
    packetType = 0;
    checksum = 0;
    csA = '0';
    csB = '0';
    rxcs = 0;
    checkfail = 0;
    
    hasReceivedData = false;
    
    currentSatsInView = 0;
    currentSatsTracked = 0;
    
    currentFixQual = 0;
    
    currentSpeed = 0.0;
    currentHeading = 0.0;
    
    baud(baudRate);
    attach(this, &CelestialReachGPS::irq_rx);
}