/**
 *  Dynamic Lighting Lib. of 7Seg-LED (add DP) x4.
 *
 *  Features:
 *   - Unified Anode/Cathode common 7Segs, and NPN/PNP switchings.
 *   - Using BusOut(s) of Digits(0-3) and Segments(A-G,DP).
 *   - Simple Interface of Class.
 *   - Non-use wait(); Useing Interrupt Timer: Ticker and Timeut.
 *   - Modulates Brightness; You Can Control of Periodic-Time and Duty.
 *   - Display Hex(0-9,A-F) of 4 Digit; Using char[4].
 */

/** @code
#include "mbed.h"
 
#include "LED7segX4.h"
 
// port make.
BusOut pDgit(p21, p22, p23, p24);   // D1, D2, D3, D4
BusOut pSeg(p11, p12, p13, p14, p15, p16, p17, p18); // A, B, C, D, E, F, G, DP

// Make instance of 7Seg-class.
LED7segX4 led7seg(pDgit, pSeg);

// for Operation check.
DigitalOut myled(LED1);

int main() {
    
    char val[4];
    char dp;
    int duty;
    
     // In this case: Anode Common, PNP swiching for Dig-common,
    // and sink port(BusOut) for seg.
    led7seg.init(true, true, 1000, 40);
    
    val[0]= 0;
    val[1]= 0;
    val[2]= 0;
    val[3]= 0;
    
    led7seg.set7seg(val);
    led7seg.setDP(0b0001);
    
    while(true) {
        myled = !myled;
        wait_ms(30);

        if(++duty > 100)
            duty= 30;
            
        val[3] += 1;
        for(int i= 0; i < 3; i++){
            if(val[3-i] > 15){
                if(i < 3){
                    val[3-i]= 0;
                    val[2-i] += 1;
                }
                dp += 1;
            }
        }
        if(val[0]==15 && val[1]==15 && val[2]==15 && val[3]==15){
            // = FFFF
            val[0]= 0;
            val[1]= 0;
            val[2]= 0;
            val[3]= 0;
        }
        if(dp > 0b1111)
            dp= 0;
        
        led7seg.set7seg(val);
        led7seg.setDP(dp);
        led7seg.setPWM(1000, duty);
        
    }
    
    return 0;
}
 * @endcode
 */
 
#pragma once

#include "mbed.h"

class LED7segX4{
public:

    /** Constructor
     *  @param BusOut; _portDig: Bus4 of Digit(3~0)
     *  @param BusOut; _portSeg: Bus8 of Segment(a~g+dp)
     */
    LED7segX4(BusOut &portDig, BusOut &portSeg);

    ~LED7segX4();
    
    
    /** Initialize
     *  @param bool; dig: Reverce(H/L) Digit(3~0)
     *  @param bool; seg: Reverce(H/L) Segment(a~g+dp)
     *  @param int; timePeriod[us]: time of priend in PWM.
     *  @param int; duty[%]: duty of PWM.
     */
    void init(bool revDig= false, bool revSeg= false, int timePeriod= 1000, int duty= 80);

    
    /** set seg TYPE;
     *  @param bool; dig: Digit(3~0)
     *  @param bool; seg: Segment(a~g+dp)
    */
    void setReverse(bool dig, bool seg);
    
    /** Set 7seg.
     *  @param chr[4]; D3~D1, 0x00(0)~0x15(F) and 0x16(OFF/NULL).
    */
    void set7seg(char chr[]);
    
    /** Set DotPoint.
     *  @param chr; 0b0000~0b1111, bit0~3; TRUE=1.
    */
    void setDP(char chr);
    
    /** Set PWM (Britness).
     *  @param int; periodic time [us]. "for Each Digit."
     *  @param int; duty ratio;1~100[%] in timePeriod.
     *  ex) In case of 1,000us and 60%, 
    */
    void setPWM(int timePeriod, int duty);
    
    
    
private:

    // ********************** Variable ******************************
    
    // pin/port set
    BusOut &_portDig;
    BusOut &_portSeg;
    
    // TimerIRQ 4 PWM
    Ticker ticker;      // Duty : Bright
    Timeout timeout;    // Duty : OFF
    float _dutySec;     // Duty(Bright) Time-width [s], in 1 period.

    // Define
    char _def7seg[17];   // Define Segment; 0~f + 16:OFF(Null)


    // Local Hold Val
    char _seg7[5];      // 4+Null           ex) 2.3E5 -> 2, 3, 14, 5
    char _dp;           // 0b0000~0b1111;   ex) 12.3E. -> 0b0101
    bool _revDig;       // REVERSE H/L 4 Transistor swDigit
    bool _revSeg;       //                      AND swSeg
    int _currentDigit;  // current digit-Num (0~3). This shifts in timeperiod of PWM.
    

    // ********************** Internal Function *********************
    void init_def7seg();    // Set/Define of seg pattern.
    
    // For DUTY of PWM. ON(Shift) and OFF.
    void OnSet();
    void OffSet();
    
};

// EOF