#include "mbed.h"
#include "BLE.h"

BLEDevice  ble;

/** サービス・キャラクタリスティックの定義 **/
// サービスのUUID
const uint8_t uuid_service1_a[16] = {0x2c,0x81,0x00,0x01,0x29,0x49,0x42,0xe2,0x83,0x68,0x75,0x64,0x6a,0xb7,0xe1,0x34};
 // キャラクタリスティックのUUID
const uint8_t uuid_char1_a[16]    = {0x2c,0x81,0x00,0x02,0x29,0x49,0x42,0xe2,0x83,0x68,0x75,0x64,0x6a,0xb7,0xe1,0x34};
const UUID uuid_service1(uuid_service1_a);
const UUID uuid_char1(uuid_char1_a);
const int char1_len = 12; //キャラクタリスティック長さ
uint8_t   char1_val[char1_len] = {0};
GattCharacteristic char1 (  uuid_char1, //UUID
                            (uint8_t *)char1_val, //初期値
                            char1_len,  //初期データ長
                            char1_len,  //最大データ長
                            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);  //プロパティ
GattCharacteristic *characteristics[] = {&char1, };
int char_num = sizeof(characteristics) / sizeof(GattCharacteristic *);
GattService        service1(  uuid_service1,    //UUID
                              characteristics,  //キャラクタリスティックの集合
                              char_num ); //キャラクタリスティック数

/* HMC5883L I2C通信用変数定義 */
// I2C設定。SDA＝ピン10、SCL＝ピン8
I2C i2c(p10, p8); 
//センサ１ ON/OFF制御＝ピン28                
DigitalOut ctrlPin1(p28); 
//センサ２ ON/OFF制御＝ピン29
DigitalOut ctrlPin2(p29); 
// HCM5883 I2C Address
const int i2cAddr = 0x3C; 

/* 指定番号のセンサを有効にする */
void enableSensor(int num){
    if (num==1){
        ctrlPin1 = 1;
        ctrlPin2 = 0;
    }else{
        ctrlPin1 = 0;
        ctrlPin2 = 1;
    }
}

/* センサを初期化する */
void initSensor(int num){
    // 有効センサの指定
    enableSensor(num);
    /* モードレジスタ設定              */
    /* Continuous-Measurement にする */
    char cmd[2];
    // モードレジスタのアドレス
    cmd[0] = 0x02; 
    // モードレジスタに書き込む値。
    cmd[1] = 0x00; 
    // コマンド送信
    i2c.write(i2cAddr, cmd, 2); 
}

/* センサ値を取得する */
void readSensor(char* buf, int num){
    // 有効センサの指定
    enableSensor(num);
    /* センサ値取得  */
    char cmd[1];
    //センサ値取得用レジスタのアドレス
    cmd[0] = 0x3; 
    //コマンド送信
    i2c.write(i2cAddr, cmd, 1); 
    //6バイト分データを読み取る
    //X,Z,Y軸の順に各2バイト
    i2c.read(i2cAddr|1, buf, 6); 
}

/* BLE接続切断時のコールバック */
void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason){
    ble.startAdvertising();
}

/* BLE接続時のコールバック */
void onConnectionCallback(const Gap::ConnectionCallbackParams_t *params){
}

/* BLE READ要求受信時のコールバック */
void onDataReadCallback(const GattReadCallbackParams *eventDataP){
}


/* タイマーによるコールバック関数 */ 
void periodicCallback(void){
    // ２つのセンサ値を取得
    readSensor((char*)&(char1_val[0]), 1);
    readSensor((char*)&(char1_val[6]), 2);
    // キャラクタリスティック値の更新
    GattAttribute::Handle_t handle = 
          char1.getValueAttribute().getHandle();
    ble.updateCharacteristicValue(
          handle,  //対象のキャラクタリスティックのハンドル
          (uint8_t *)&char1_val,//更新後のデータ
          sizeof(char1_val));  //データ長
}

/* main */
int main(void){
    
    // I2C初期化
    initSensor(1);
    initSensor(2);

    /*  BLE初期化 */
    ble.init();
    ble.onConnection(onConnectionCallback);
    ble.onDisconnection(disconnectionCallback);
    ble.onDataRead(onDataReadCallback);

    /*　 アドバタイズ設定 */
    //Centralからのアドバタイズメントパケットの受信可否の設定
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE); 
    // パケットに含めるデバイス名称設定
    const char  device_name[] = "BLE Nano";
    ble.accumulateAdvertisingPayload( GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)device_name, sizeof(device_name)); 
    // Centralからの接続可を示す設定
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    // アドバタイズ間隔設定。 0.625msの倍数を指定する。ここでは100ms.
    ble.setAdvertisingInterval(160); 
    //アドバタイズ開始
    ble.startAdvertising(); 

    /* サービス登録 */
    ble.addService(service1);

    /*  タイマー設定 */
    Ticker ticker;
    ticker.attach(
           periodicCallback, //コールバックされる関数
           1 );   //コールバック間隔[秒]

    /*  メインループ*/
    while (true) {
        ble.waitForEvent();
    }
}