/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#ifndef AJK_TON_H
#define AJK_TON_H

#include "mbed.h"

namespace AjK {

/** @defgroup API The Ton API */

/** Ton module
 *
 * @author Andy Kirkham
 * @see API
 * @see example1.h
 * @see example2.h
 *
 */
class Ton {

protected:
    bool    _lastInput;
    int     _counter;
    int     _reload;
    int     _mode;
    Ticker  _tick;
    
    /** init
     * Used to init the properties.
     */
    void init(void) {
        _lastInput = false;
        _mode      = InputResets;
        _tick.attach_us(this, &Ton::ticktock, 1000);
    }
    
public:
    friend class Ticker;
    
    enum Ops { On = 1, Off = 0 };
    
    /** Mode
     */
    enum Mode {
        InputResets = 0,    /*!< Loss of input resets the timer. */
        InputPauses         /*!< Loss of input pauses the timer. */
    };
    
    /** ticktock
     * Used by the Ticker object to call to once per millisecond.
     */
    void ticktock(void) {
        switch (_mode) {
        case InputResets:
            if (_lastInput) {
                if (_counter > 0) _counter--;
            }
            else {
                _counter = _reload;
            }
            break;
        case InputPauses:
            if (_lastInput) {
                if (_counter > 0) _counter--;
            }
            break;
        }
    }
    
    /** isRunning
     * Is the timer active?
     * @return true if active, false otherwise.
     */
    bool isRunning(void) {
        bool b = false;
        switch (_mode) {
        case InputResets:
            b = _lastInput;
            break;
        case InputPauses:
            b = _counter > 0;
        }
        return b;
    }
    
    /** setTime
     * Used to set the timer value, in milliseconds.
     * @ingroup API
     * @param int i The value to time.     
     */
    void setTime(int i) { _counter = _reload = i; }
    
    /** getTime
     * Used to get the timer value, in milliseconds.
     * @ingroup API
     * @return int The value to time.     
     */
    int getTime(void) { return _reload; }
    
    /** getTimeLeft
     * Used to get the timer value remaining, in milliseconds.
     * @ingroup API
     * @return int The time remaining to zero (timeout). 
     */
    int getTimeLeft(void) { return _reload - _counter; }
    
    /** setMode
     * Set the Ton operational mode.
     * Valid values are Ton::InputResets or Ton::InputPauses.
     * The default mode is Ton::InputPauses
     * @ingroup API
     * @see Mode
     * @param enum Mode
     */
    void setMode(Mode i) { _mode = i; }
    
    /** reset
     * Resets the timer to the default value.
     * @ingroup API
     */
    void reset(void) { _counter = _reload; }
    
    /** reset
     * Overloaded method, reset the timer to the value supplied.
     * @ingroup API
     * @param int i The time to count.
     */
    void reset(int i) { _counter = _reload = i; }
    
    /** Constructor
     */
    Ton(int to, int mode = InputResets) { init(); _counter = _reload = to; _mode = mode; }
    
    /** Destructor
     */
    ~Ton() { _tick.detach(); }
    
    /** operator=
     * Controls the starting of the timer.
     * @ingroup API
     */
    Ton& operator= (int value) {
        _lastInput = value ? true : false;
        if (_mode == InputResets && !_lastInput) _counter = _reload; 
        return *this;
    }
    
    /** operator int()
     * @ingroup API
     * @return zero if not timed out yet, non-zero otherwise.
     */
    operator int() {
        return (_counter == 0) ? 1 : 0;
    }
        
};

}; // namespace AjK ends.

using namespace AjK;

#endif
