/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/


#include "mbed.h"
#include "SimpleTLE5206.h"

namespace AjK {

SimpleTLE5206::SimpleTLE5206(SimpleTLE5206Output *in1, SimpleTLE5206Output *in2)
{
    _in1 = in1;
    _in2 = in2;
    init(0);
}

SimpleTLE5206::SimpleTLE5206(SimpleTLE5206Output *in1, SimpleTLE5206Output *in2, int duty_cycle_hz)
{
    _in1 = in1;
    _in2 = in2;
    if (duty_cycle_hz == 0) error("Division by zero!\n");
    init((uint32_t)(12000000UL / duty_cycle_hz * 2));
}

void
SimpleTLE5206::init(uint32_t duty)
{
    _in1->as_pwm();
    _in2->as_pwm();
    if (duty != 0) setDuty(duty);
    setSpeed(0.0);
}

void
SimpleTLE5206::setDuty(uint32_t duty)
{

    _duty = duty;
    
    // Skip the setup if the duty has already been set
    // by a previous instance of controller.
    if (LPC_PWM1->MR0 == duty) {
        return;
    }
    
    // Ensure powered up (default is 1)  
    LPC_SC->PCONP |= (1UL << 6);  
        
    // CCLK/4 = 24MHz
    LPC_SC->PCLKSEL0 &= ~(3UL << 12);
    
    // Reset.
    LPC_PWM1->TCR   = 2; 
    
    LPC_PWM1->PR    = 0;
    LPC_PWM1->MR0   = _duty;
    LPC_PWM1->MR1   = 0;
    LPC_PWM1->MR2   = 0;
    LPC_PWM1->MR3   = 0;
    LPC_PWM1->MR4   = 0;
    LPC_PWM1->MR5   = 0;
    LPC_PWM1->MR6   = 0;
        
    LPC_PWM1->MCR   = 2; // MR0 resets TC.        
    LPC_PWM1->TCR   = 9; // Enable.        
}

void
SimpleTLE5206::setSpeed(double speed)
{
    uint32_t value1, value2;

    // Ensure we cap the speed to +/-1.0
    if (speed > +1.0) speed = +1.0;
    if (speed < -1.0) speed = -1.0;

    value1 = _duty + 1; // Output always on.
    value2 = _duty + 1; // Output always on.
        
    if (speed != 0.0) {
        if (speed > 0.0) {
            value2 = (uint32_t)(speed * _duty);       // Scale for requested speed.
            if (value2 >= _duty) value2 = _duty - 1;  // Don't allow the value to overrun the duty.
            value2 = _duty - value2;                  // Invert logic sense.
        }
        
        if (speed < 0.0) {
            speed *= -1; // invert sign.              
            value1 = (uint32_t)(speed * _duty);       // Scale for requested speed.
            if (value1 >= _duty) value1 = _duty - 1;  // Don't allow the value to overrun the duty.
            value1 = _duty - value1;                  // Invert logic sense.
        }
    }
    
    switch(_in1->getPin()) {
        case p21:            setMRx(Pwm6, value1); break;
        case p22:            setMRx(Pwm5, value1); break;
        case p23: case LED4: setMRx(Pwm4, value1); break;
        case p24: case LED3: setMRx(Pwm3, value1); break;
        case p25: case LED2: setMRx(Pwm2, value1); break;
        case p26: case LED1: setMRx(Pwm1, value1); break;                          
    }
        
    switch(_in2->getPin()) {
        case p21:            setMRx(Pwm6, value2); break;
        case p22:            setMRx(Pwm5, value2); break;
        case p23: case LED4: setMRx(Pwm4, value2); break;
        case p24: case LED3: setMRx(Pwm3, value2); break;
        case p25: case LED2: setMRx(Pwm2, value2); break;
        case p26: case LED1: setMRx(Pwm1, value2); break;                          
    }
}

void 
SimpleTLE5206::setMRx(PwmCh ch, uint32_t value) 
{
    switch(ch) {
        case Pwm1: LPC_PWM1->MR1 = value; break;
        case Pwm2: LPC_PWM1->MR2 = value; break;
        case Pwm3: LPC_PWM1->MR3 = value; break;
        case Pwm4: LPC_PWM1->MR4 = value; break;
        case Pwm5: LPC_PWM1->MR5 = value; break;
        case Pwm6: LPC_PWM1->MR6 = value; break;        
    }
    LPC_PWM1->LER |= (1UL << ch);
}

}; // namespace AjK ends.
