/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/


#ifndef AJK_SIMPLETLE5206PROFILER_H
#define AJK_SIMPLETLE5206PROFILER_H

#include "mbed.h"
#include "SimpleTLE5206.h"

namespace AjK {

class SimpleTLE5206Profiler : public SimpleTLE5206 {

protected:
    Ticker  _poll;
    double  _desired_speed;
    double  _actual_speed;
    double  _accel_rate;
    double  _decel_rate;
    int     _poll_interval;

    void poll(void) 
    {
        if (_actual_speed < _desired_speed) {
            _actual_speed += _accel_rate;
            if (_actual_speed > _desired_speed) {
                _actual_speed = _desired_speed;
            }
            SimpleTLE5206::setSpeed(_actual_speed);
        }
        if (_actual_speed > _desired_speed) {
            _actual_speed -= _decel_rate;
            if (_actual_speed < _desired_speed) {
                _actual_speed = _desired_speed;
            }
            SimpleTLE5206::setSpeed(_actual_speed);
        }
    }
    
public:

    friend class Ticker;
    
    SimpleTLE5206Profiler(SimpleTLE5206Output *in1, SimpleTLE5206Output *in2, int duty_cycle_hz) : 
        SimpleTLE5206(in1, in2, duty_cycle_hz) 
    {
        _desired_speed = 0.0;
        _actual_speed  = 0.0;
        _accel_rate = 0.01;
        _decel_rate = 0.01;
        _poll_interval = 10000; // 10ms
        _poll.attach_us(this, &SimpleTLE5206Profiler::poll, _poll_interval);   
    }
    
    int  getPollInterval(void)  { return _poll_interval; }
    void setPollInterval(int i) 
    { 
        _poll_interval = i;
        _poll.detach();
        _poll.attach_us(this, &SimpleTLE5206Profiler::poll, _poll_interval);
    }
    
    void setAccelRate(double rate) { _accel_rate = rate; } 
    double getAccelRate(void) { return _accel_rate; } 
    
    void setDecelRate(double rate) { _decel_rate = rate; } 
    double getDecelRate(void) { return _decel_rate; } 
    
    void setSpeed(double demand_speed)
    {
        _desired_speed = demand_speed;
    }
    
    double getSpeed(void) { return _actual_speed; }
    
    void eStop(void) 
    {
        _desired_speed = _actual_speed = 0.0;
        SimpleTLE5206::setSpeed(_desired_speed);
    }
    
   
};

}; // namespace AjK ends.


#endif
