
#include "mbed.h"
#include "SimpleTLE5206.h"

/* NOTE! The SimpleTLE5206 library supports the TLE5206 in 
 * Mode 2 Sign/Magnitude Control using two PWM outputs to
 * control speed and direction. 
 *
 * Pins that can be used are p21, p22, p23, p24, p25 and/or p26
 * in pairs. So the library supports upto 3 TLE5206 devices/motors.
 *
 * All PWM outputs use a common duty cycle. Therefore the third arg
 * to the constructor must be the same for all TLE5206 devices. To
 * ensure this, we use a #define DUTY_CYCLE_IN_HERTZ and supply it
 * to all instances of controllers created.
 *
 * Additionally you can use LED1, LED2, LED3 and.or LED4 as mimics.
 * However, if using:-
 *  LED1 you cannot use p26 to drive an in to TLE5206
 *  LED2 you cannot use p25 to drive an in to TLE5206
 *  LED3 you cannot use p24 to drive an in to TLE5206
 *  LED4 you cannot use p23 to drive an in to TLE5206
 *
 * The function SimpleTLE5206::setSpeed() takes a single arg, a double,
 * and should be in the range +1.0 to -1.0 where +1.0 is full speed in
 * a CW direction, -1.0 is full speed in a CCW direction and 0 is stopped.
 */

#define DUTY_CYCLE_IN_HERTZ 50

// Create a motor "A", driven by a TLE5206 on pins 21 and 22 (attach scope first, not a motor!)
SimpleTLE5206Output Ain1(p21);    // TLE5206 In1 is connected to p21
SimpleTLE5206Output Ain2(p22);    // TLE5206 In2 is connected to p22
SimpleTLE5206 motorA(&Ain1, &Ain2, DUTY_CYCLE_IN_HERTZ); // Create the TLE5206 controller.

// Create a motor "B", driven by a TLE5206 but on LEDs as a mimic.
SimpleTLE5206Output Bin1(LED1);   // TLE5206 In1 is connected to LED1
SimpleTLE5206Output Bin2(LED2);   // TLE5206 In2 is connected to LED2
SimpleTLE5206 motorB(&Bin1, &Bin2, DUTY_CYCLE_IN_HERTZ); // Create the TLE5206 controller.

// Create a motor "C", driven by a TLE5206 but on LEDs as a mimic.
SimpleTLE5206Output Cin1(LED3);   // TLE5206 In1 is connected to LED3
SimpleTLE5206Output Cin2(LED4);   // TLE5206 In2 is connected to LED4
SimpleTLE5206 motorC(&Cin1, &Cin2, DUTY_CYCLE_IN_HERTZ); // Create the TLE5206 controller.

Ticker A, B, C;

volatile double demand[3];
volatile double speed[3];

#define PI 3.14159265

void Acallback(void) {
    speed[0] = sin(demand[0] * PI / 180.0);
    if (++demand[0] >= 360.0) demand[0] = 0.0;
    motorA.setSpeed(speed[0]);
}

void Bcallback(void) {
    speed[1] = sin(demand[1] * PI / 180.0);
    if (++demand[1] >= 360.0) demand[1] = 0.0;
    motorB.setSpeed(speed[1]);
}

void Ccallback(void) {
    speed[2] = sin(demand[2] * PI / 180.0);
    if (++demand[2] >= 360.0) demand[2] = 0.0;
    motorC.setSpeed(speed[2]);
}

int main() {

    volatile int trash = 0;

    motorA.setSpeed(0);
    motorB.setSpeed(0);
    motorC.setSpeed(0);
    
    // Init the global variables.
    for (int i = 0; i < 3; i++) {
        demand[i] = speed[i] = 0.0;
    }
    
    // Note, you probably wouldn't want to move the speed of
    // a motor at this rate, may break it. This example uses 
    // a high update rate (0.025) assuming you are attaching 
    // an oscilloscope just for testing. It goes without saying
    // that the update rates for B and C are way to big, I just 
    // choose these (0.005 and 0.0025) because it looks pretty
    // when used on LEDs!
    // Always use appropriate accel/decel rates when handling 
    // motors/external hardware that moves. See example2.h for
    // a simple linear accel/decel profiler that does this.
    
    A.attach(Acallback, 0.025);
    B.attach(Bcallback, 0.005);
    C.attach(Ccallback, 0.0025);
    
    while(1) {    
        /* The main loop has little to do as the Ticker callbacks
           set-up the speed changes for the example. So give it something 
           to do. Maybe change this and use the spare time to calculate PI 
           more accurately? Lol, just kidding. */
        trash++;
    }
}


