/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
    
    @file          example3.h 
    @purpose       Simple round-robin cooperative scheduler example
    @date          Mar 2011
    @author        Andy Kirkham
*/

/*
    SimpleScheduler is a cooperative scheduler. What that means is YOU, the
    programmer MUST cooperate with it. SimpleScheduler will call your task
    functions based on the schedule you provide. However, it's not a guarenteed
    scheduled time. It's on a "best effort". However, no tasks are called until
    the currently executing task completes. What that means is your task function
    must return before another task can be called.
    
    Your tasks therefore should execute as fast as possible and return as soon as
    they can. But when Mbed's wait API is used you stall your entire system. This
    example shows how you can wait() in a task without actually waiting at all. In
    order to do this we use the "Ton" (Timer ON) library. You will need to import
    this library into your project to use this example:-
    
    http://mbed.org/users/AjK/libraries/Ton/latest
    
    What follows is basically example1. However, we are going to flash LED1 by 
    "waiting" 1second in the task. 
    
*/

#include "mbed.h"
#include "SimpleScheduler.h"
#include "Ton.h"

Ton t1(1000); // Timer ON delay of one second.

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);

void f1(SimpleTask *task) {
    // wait(1);     
    if (!t1.isRunning()) {
        t1 = Ton::On;
        // Code that should execute once before the wait() goes here.
        // ...
    }        
    if (t1) {
        // This code will be executed once the timer times out.
        led1 = !led1; 
        t1.reset();
    }
}
void f2(SimpleTask *task) { led2 = !led2; }
void f3(SimpleTask *task) { led3 = !led3; }
void f4(SimpleTask *task) { led4 = !led4; }

SimpleScheduler *scheduler;

int main() {

    scheduler = new SimpleScheduler;
    
    scheduler
        ->addTask( new SimpleTask(0,   f1) )
        ->addTask( new SimpleTask(200, f2) )
        ->addTask( new SimpleTask(300, f3) )
        ->addTask( new SimpleTask(0.4, f4) )
    ;
    
    scheduler->run();    
}

