/*
    Copyright (c) 2011 Andy Kirkham
 
    Permission is hereby granted, free of charge, to any person obtaining a copy
    of this software and associated documentation files (the "Software"), to deal
    in the Software without restriction, including without limitation the rights
    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
    copies of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
    THE SOFTWARE.
*/

#ifndef AJK_SIMPLERCSERVOS_H
#define AJK_SIMPLERCSERVOS_H

#include "LPC17xx.h"
#include "SimpleRCmbed.h"

namespace AjK {

class SimpleRCservos {

public:
    enum Servo { NoServo0, Servo1, Servo2, Servo3, Servo4, Servo5, Servo6, NumOfServos  };
    
protected:
    PinName _pinName;
    
    uint32_t _duty;
    
    uint32_t _mid[NumOfServos];
    
    double _limitMax[NumOfServos];
    
    double _limitMin[NumOfServos];
    
    uint32_t _msMin[NumOfServos];
    
    uint32_t _msMax[NumOfServos];
            
    void init(uint32_t duty);
       
    void setRangeMin(Servo ch, uint32_t u) { _msMin[ch] = u * 24; }
    
    void setRangeMax(Servo ch, uint32_t u) { _msMax[ch] = (u * 24) - _msMin[ch]; }
        
    void setMRx(Servo ch, uint32_t value);
    
    double limit(Servo ch, double degrees);
                
public:

    /** Constructor
     */
    SimpleRCservos() { init(480000UL); setDuty(); }

    /** setDuty
     * 
     * Set's the duty, or period, of the PWM pulses.
     * The default is 20ms and the parameter passed in is the 
     * number of "clock ticks" that represents the period.
     * 480000 gives rise to 20ms by default. It's rare that
     * you would alter this value. Remember, with the PWM
     * peripheral there is only a single, common,  period for 
     * all six PWM outputs.
     *
     * @param duty The number of "clock ticks" for the duty cycle.
     */
    void setDuty(uint32_t duty = 480000);
    
    /** enable1
     * 
     * Enable PWM channel 1.
     *
     * @param PinName p26(P2_0) or LED1(P1_18)
     */
    void enable1(PinName pin = p26);
    
    /** enable2
     * 
     * Enable PWM channel 2.
     *
     * @param PinName p25(P2_1) or LED2(P1_20)
     */
    void enable2(PinName pin = p25);
    
    /** enable3
     * 
     * Enable PWM channel 3.
     *
     * @param PinName p24(P2_2) or LED3(P1_21)
     */
    void enable3(PinName pin = p24);
    
    /** enable4
     * 
     * Enable PWM channel 4.
     *
     * @param PinName p23(P2_3) or LED4(P1_23)
     */
    void enable4(PinName pin = p23);
    
    /** enable5
     *
     * Enable PWM channel 5.
     *
     * @param PinName p22(P2_4) or P1_24
     */
    void enable5(PinName pin = p22);
    
    /** enable6
     *
     * Enable PWM channel 6.
     *
     * @param PinName p21(P2_5) or P1_26
     */
    void enable6(PinName pin = p21);
    
    /** enable
     *
     * Enable the specified channel. The default pin on port 2 is assumed.
     *
     * @param Servo The channel to enable, Servo1 to Servo6
     */
    void enable(Servo ch);
    
    /** setLimitMin
     * 
     * Set the minimum logical limit for the specified channel.
     *
     * @param Servo channel
     * @param double The lower logical limit.
     */
    void setLimitMin(Servo ch, double d) { _limitMin[ch] = d; }
    
    /** getLimitMin
     * 
     * Get the minimum logical limit for the specified channel.
     *
     * @param Servo channel
     * @return double The lower logical limit.
     */
    double getlimitMin(Servo ch)   { return _limitMin[ch]; }
    
    /** setLimitMax
     * 
     * Set the maximum logical limit for the specified channel.
     *
     * @param Servo channel
     * @param double The upper logical limit.
     */
    void setLimitMax(Servo ch, double d) { _limitMax[ch] = d; }
    
    /** getLimitMax
     * 
     * Get the maximum logical limit for the specified channel.
     *
     * @param Servo channel
     * @return double The upper logical limit.
     */
    double getLimitMax(Servo ch)   { return _limitMax[ch]; }
    
    /** setLimits
     *
     * Set the logical upper and lower limits for channel.
     *
     * @param Servo channel
     * @param double The minimum logical limit
     * @param double The maximum logical limit
     */
    void setLimits(Servo ch, double min, double max) { 
        _limitMin[ch] = min; 
        _limitMax[ch] = max; 
    }
    
    /** getRangeMin
     * 
     * Set the minimum physical limit for the specified channel.
     *
     * @param Servo channel
     * @param double The lower phyiscal limit in microseconds
     */
    uint32_t getRangeMin(Servo ch, uint32_t u) { return _msMin[ch] / 24; }
    
    /** getRangeMax
     * 
     * Set the maximum physical limit for the specified channel.
     *
     * @param Servo channel
     * @param uint32 The upper phyiscal limit in microseconds
     */
    uint32_t getRangeMax(Servo ch, uint32_t u) { return (_msMax[ch] / 24) + _msMin[ch]; }
    
    /** setRange
     * 
     * Set the minimum and maximum physical limits for the specified channel.
     *
     * @param Servo channel
     * @param uint32 The lower phyiscal limit in microseconds
     * @param uint32 The upper phyiscal limit in microseconds
     */
    void setRange(Servo ch, uint32_t min, uint32_t max) { 
        setRangeMin(ch, min);
        setRangeMax(ch, max);
    }
    
    /** position
     *
     * Set the desired position of the servo.
     * This value should be between the lower and upper logical limits.
     *
     * Under normal circumstances this fuction returns the same position is 
     * was supplied. However, if the supplied position is outside the logical
     * system limits we return the value we actually went to. This is almost
     * always the value of the logical limit that is set for that direction.
     *
     * @param Servo channel
     * @param double the logical position to move to.
     * @return double The actual calculated position moved to.
     */
    double position(Servo ch, double degrees);
    
    /** neutral
     *
     * Tell the servo to move to it's neutral position (1.5ms pulse width)
     *
     * @param Servo channel
     */
    void neutral(Servo ch);
    
}; // class RCservo ends.

}; // namespace AjK ends.

using namespace AjK;

#endif /* AJK_SIMPLERCSERVOS_H */
